import builtins
from collections.abc import Collection
from inspect import getmodule, isclass, isfunction, ismethod, ismodule
from types import ModuleType
from typing import Any

from atoti_core import get_top_level_package_name

_BUILTIN_TYPES: Collection[Any] = {
    getattr(builtins, name)
    for name in dir(builtins)
    if isinstance(getattr(builtins, name), type)
}

_NON_PERSONAL_PACKAGES = ("atoti", "atoti_core", "atoti_query")

_PERSONAL_TYPE_NAME = "__REDACTED_TYPE__"


def _is_non_personal_module(module: ModuleType, /) -> bool:
    return get_top_level_package_name(module.__name__) in _NON_PERSONAL_PACKAGES


def get_non_personal_type_name(value: Any, /) -> str:
    module = getmodule(value)
    is_value = not (
        ismodule(value) or isclass(value) or isfunction(value) or ismethod(value)
    )
    value_type = type(value) if is_value else value
    return (
        f"{f'{module.__name__}.{value_type.__name__}' if is_value else value_type}"
        if module and _is_non_personal_module(module)
        else f"{value_type.__name__ if value_type in _BUILTIN_TYPES else _PERSONAL_TYPE_NAME}"
    )
