from typing import Optional, Union, overload

from atoti_core import doc

from .._measure_convertible import NonConstantMeasureConvertible
from .._measure_description import MeasureDescription
from ..column import Column
from ..scope._scope import Scope
from ._agg import agg
from ._utils import SCOPE_DOC


@overload
def single_value(operand: Column, /) -> MeasureDescription:
    ...


@overload
def single_value(
    operand: NonConstantMeasureConvertible, /, *, scope: Scope
) -> MeasureDescription:
    ...


@doc(scope=SCOPE_DOC)
def single_value(
    operand: Union[Column, NonConstantMeasureConvertible],
    /,
    *,
    scope: Optional[Scope] = None,
) -> MeasureDescription:
    """Return a measure equal to the value aggregation of the operand across the specified scope.

    If the value is the same for all members of the level the operand is being aggregated on, it will be propagated to the next level.

    ``None`` values are ignored: they will not prevent the propagation.

    Args:
        operand: The measure or table column to aggregate.
        {scope}

    Example:
        >>> df = pd.DataFrame(
        ...     columns=["Continent", "Country", "City", "Price"],
        ...     data=[
        ...         ("Europe", "France", "Paris", 200.0),
        ...         ("Europe", "France", "Lyon", 200.0),
        ...         ("Europe", "UK", "London", 200.0),
        ...         ("Europe", "UK", "Manchester", 150.0),
        ...         ("Europe", "France", "Bordeaux", None),
        ...     ],
        ... )
        >>> table = session.read_pandas(
        ...     df, table_name="City price table", keys=["Continent", "Country", "City"]
        ... )
        >>> cube = session.create_cube(table)
        >>> h, l, m = cube.hierarchies, cube.levels, cube.measures
        >>> geography_level_names = ["Continent", "Country", "City"]
        >>> h["Geography"] = [table[name] for name in geography_level_names]
        >>> for name in geography_level_names:
        ...     del l[name, name]
        ...
        >>> m["Price.VALUE"] = tt.agg.single_value(table["Price"])
        >>> cube.query(
        ...     m["Price.VALUE"],
        ...     levels=[l["City"]],
        ...     include_empty_rows=True,
        ...     include_totals=True,
        ... )
                                     Price.VALUE
        Continent Country City
        Total
        Europe
                  France                  200.00
                          Bordeaux
                          Lyon            200.00
                          Paris           200.00
                  UK
                          London          200.00
                          Manchester      150.00

        * The :guilabel:`City` level is the most granular level so the members have the same value as in the input dataframe (including ``None`` for Bordeaux).
        * All the cities in France have the same price or ``None`` so the value is propagated to the :guilabel:`Country` level.
          The values in the UK cities are different so :guilabel:`Price.VALUE` is ``None``.
        * The cities in Europe have different values so the :guilabel:`Price.VALUE` is ``None`` at the :guilabel:`Continent` level.

    """
    # The type checkers cannot see that the `@overload` above ensure that this call is valid.
    return agg(operand, plugin_key="SINGLE_VALUE_NULLABLE", scope=scope)  # type: ignore[arg-type] # pyright: ignore[reportGeneralTypeIssues]
