"""Student's t distribution.

For more information read:

* `scipy.stats.t <https://docs.scipy.org/doc/scipy/reference/generated/scipy.stats.t.html>`__
* `The Student's t Wikipedia page <https://en.wikipedia.org/wiki/Student%27s_t-distribution>`__

"""
from __future__ import annotations

from ..._measure_convertible import NonConstantMeasureConvertible
from ..._measure_description import MeasureDescription, convert_to_measure_description
from ..._measures.calculated_measure import CalculatedMeasure, Operator
from ._utils import NumericMeasureConvertible, ensure_strictly_positive


def pdf(
    point: NonConstantMeasureConvertible,
    /,
    *,
    degrees_of_freedom: NumericMeasureConvertible,
) -> MeasureDescription:
    r"""Probability density function for a Student's t distribution.

    The pdf of a Student's t-distribution is:

    .. math::

        \operatorname {pdf}(x)=\frac {\Gamma (\frac {\nu +1}{2})}{\sqrt {\nu \pi }\,\Gamma (\frac {\nu }{2})} \left(1+\frac {x^{2}}{\nu }\right)^{-\frac {\nu +1}{2}}

    where :math:`\nu` is the number of degrees of freedom and :math:`\Gamma` is the gamma function.

    Args:
        point: The point where the function is evaluated.
        degrees_of_freedom: The number of degrees of freedom.
            Must be positive.

    See Also:
        `The Student's t Wikipedia page <https://en.wikipedia.org/wiki/Student%27s_t-distribution>`__

    """
    ensure_strictly_positive(degrees_of_freedom, "degrees_of_freedom")
    return CalculatedMeasure(
        Operator(
            "student_density",
            [
                convert_to_measure_description(arg)
                for arg in [point, degrees_of_freedom]
            ],
        )
    )


def cdf(
    point: NonConstantMeasureConvertible,
    /,
    *,
    degrees_of_freedom: NumericMeasureConvertible,
) -> MeasureDescription:
    """Cumulative distribution function for a Student's t distribution.

    Args:
        point: The point where the function is evaluated.
        degrees_of_freedom: The number of degrees of freedom.
            Must be positive.

    See Also:
        `The Student's t Wikipedia page <https://en.wikipedia.org/wiki/Student%27s_t-distribution>`__

    """
    ensure_strictly_positive(degrees_of_freedom, "degrees_of_freedom")
    return CalculatedMeasure(
        Operator(
            "student_cumulative_probability",
            [
                convert_to_measure_description(arg)
                for arg in [point, degrees_of_freedom]
            ],
        )
    )


def ppf(
    point: NonConstantMeasureConvertible,
    /,
    *,
    degrees_of_freedom: NumericMeasureConvertible,
) -> MeasureDescription:
    """Percent point function for a Student's t distribution.

    Also called inverse cumulative distribution function.

    Args:
        point: The point where the function is evaluated.
        degrees_of_freedom: The number of degrees of freedom.
            Must be positive.

    See Also:
        `The Student's t Wikipedia page <https://en.wikipedia.org/wiki/Student%27s_t-distribution>`__

    """
    ensure_strictly_positive(degrees_of_freedom, "degrees_of_freedom")
    return CalculatedMeasure(
        Operator(
            "student_ppf",
            [
                convert_to_measure_description(arg)
                for arg in [point, degrees_of_freedom]
            ],
        )
    )
