from __future__ import annotations

from .._measure_convertible import MeasureOperation, NonConstantMeasureConvertible
from .._measure_description import convert_to_measure_description


def sqrt(measure: NonConstantMeasureConvertible, /) -> MeasureOperation:
    """Return a measure equal to the square root of the passed measure.

    Example:
        >>> df = pd.DataFrame(
        ...     columns=["City", "A", "B", "C", "D"],
        ...     data=[
        ...         ("Berlin", 15.0, 10.0, 10.1, 1.0),
        ...         ("London", 24.0, 16.0, 20.5, 3.14),
        ...         ("New York", -27.0, 15.0, 30.7, 10.0),
        ...         ("Paris", 0.0, 0.0, 0.0, 0.0),
        ...     ],
        ... )
        >>> table = session.read_pandas(df, keys=["City"], table_name="Math")
        >>> cube = session.create_cube(table)
        >>> l, m = cube.levels, cube.measures
        >>> m["√B"] = tt.math.sqrt(m["B.SUM"])
        >>> cube.query(m["B.SUM"], m["√B"], levels=[l["City"]])
                  B.SUM    √B
        City
        Berlin    10.00  3.16
        London    16.00  4.00
        New York  15.00  3.87
        Paris       .00   .00

    """
    return convert_to_measure_description(measure) ** 0.5
