import abc
from typing import Any, Optional, Callable


class CellFormatter(abc.ABC):
    """
    Subclasses of this class can be used with CSVParser to determine the output of the parser on cells in the
    CSV file. A CellFormatter object passed to the constructor of a CSVParser will be given a chance to format
    the resulting text data of each text when it is parsed by the parser. After parsing out a text, the format
    method of the parser's CellFormatter object is called and then the content is placed in the data structure that
    is returned by the CSVParser object. """

    @abc.abstractmethod
    def format(self, content: str) -> Any:
        """Receives a single text's content during parsing of a CSV in CSVParser and should return the
        the text's content post formatting. Formatting can be arbitrary.

        A class should implement this method in order to be passed to the __init__ of CSVParser. Inheriting
        this class is not required, but it is recommended

        :param content: the content of the CSV text parsed by the CSVParser
        :raises ParseError: if the method cannot parse the content string. If CSVParser sets self.debug to True,
                    then this error is caught and a new error with more information is raised `from` this error.
                    Otherwise, this error is raised unchanged
        :returns: the result of formatting the text
        """
        pass


class DefaultCellFormatter(CellFormatter):
    """
    Does not change the contents of a CSV text. Exists as the default option in CSVParser
    """

    def format(self, content: str) -> str:
        return content


class CleanedCellFormatter(CellFormatter):
    """
    Removes whitespace other than a plain space and double quotes from the content of a CSV text. Since newlines
    are supported inside of double quoted cells by the CSVParser, this will remove them among other the other things
    listed. This can be changed by passing a list of strings to be removed to the __init__ of this class when creating it
    all instances of all the strings in the list passed to __init__ will be removed from the cell's content
    """

    def __init__(self, disallowed_strings: Optional[list[str]] = None):
        if disallowed_strings is None:
            disallowed_strings = ['\r', '\n', '\t', '"']
        self.disallowed_strings = disallowed_strings

    def format(self, content: str) -> str:
        def cleaned(s: str) -> str:
            for string in self.disallowed_strings:
                s = s.replace(string, '')
            return s

        return cleaned(content)


class ClearInvalidCellFormatter(CellFormatter):
    """
    Clears the content of all cells which do not meet a certain criteria. The is_valid parameter is the predicate
    which determines what content is valid. If is_valid returns False the content of the cell is cleared but the cell
    and column are not deleted. If is_valid returns True the content of a cell is unchanged
    """
    def __init__(self, is_valid: Callable[[str], bool]):
        self.is_valid = is_valid
        super(ClearInvalidCellFormatter, self).__init__()

    def format(self, content: str) -> str:
        return content if self.is_valid(content) else ''