from PyQt5.QtWidgets import QApplication, QWidget, QMainWindow, QVBoxLayout, QMessageBox, QFileDialog, QLineEdit, QComboBox, QSystemTrayIcon
from PyQt5 import uic, QtCore, QtGui, QtWidgets
from PyQt5.QtGui import QIcon, QDoubleValidator, QIntValidator
from matplotlib.widgets import Cursor
import matplotlib.pyplot as plt
from matplotlib.backends.backend_qt5agg import FigureCanvasQTAgg, NavigationToolbar2QT
from matplotlib.figure import Figure
from matplotlib.ticker import FormatStrFormatter
from matplotlib import cm
from mpl_toolkits.axes_grid1 import make_axes_locatable
import numpy as np
from math import floor
import pathlib
import json
import os
import webbrowser
import tempfile
import datetime

import transistordatabase as tdb
import buck_converter_functions
import boost_converter_functions
import buck_boost_converter_functions
import comparison_tools_functions



def resource_path(relative_path):
    try:
        base_path = sys._MEIPASS
    except Exception:
        base_path = os.path.abspath(".")

    return os.path.join(base_path, relative_path)
    
class MainWindow(QMainWindow):
    """
    MainWindow class which inherits from QMainWindow and contains all the code and functions to implement all
    functionalities into the code generated by QTDesigner in MainWindow.ui
    """
    def __init__(self):
        super(MainWindow, self).__init__()

        MainWindow_path=os.path.join(os.path.dirname(__file__),"MainWindow.ui")
        uic.loadUi(MainWindow_path, self)

        sys.stdout = EmittingStream(text_written=self.standard_output_written)

        self.tray_icon = QSystemTrayIcon(QIcon(resource_path("window_icon.png")), parent=app)
        self.tray_icon.setToolTip("Transistordatabase")
        self.tray_icon.show()
        self.tray_icon.activated.connect(self.show)

        self.setWindowTitle(f"Transistordatabase - {tdb.__version__}")

        tdb.update_from_fileexchange(overwrite=True)

        transistor_list = self.get_transistor_list()

        self.setWindowIcon(QtGui.QIcon("window_icon"))

        self.translation_dict = {
            'IEC 60747-9': 'IEC 60747-9 (MOSFET)',
            'IEC 60747-8': 'IEC 60747-8 (IGBT)',
            'Mitsubishi': 'Mitsubishi',
            'Infineon': 'Infineon',
            'Wolfspeed': 'Wolfspeed',

        }

        self.action_save_settings.triggered.connect(self.save_settings)
        self.action_export_settings.triggered.connect(self.export_settings)
        self.action_load_settings.triggered.connect(self.load_settings)
        self.action_stdout.triggered.connect(self.show_stdout)
        self.action_clear_comparison_tools.triggered.connect(self.clear_comparison_tools)
        self.action_clear_topology_calculator.triggered.connect(self.clear_topology_calculator)
        self.action_clear_create_transistor.triggered.connect(self.clear_create_transistor)
        self.action_exit.triggered.connect(app.exit)

        ### Actions for 'Transistor' ###
        self.action_delete_transistor.triggered.connect(self.delete_marked_transistor_search_database_from_local_tdb)
        self.action_show_original_datasheet.triggered.connect(self.webbrowser_original_datasheet)
        self.action_show_virtual_datasheet.triggered.connect(self.webbrowser_virtual_datasheet)
        self.action_add_transistor_to_transistordatabase_file_exchange.triggered.connect(self.email_add_transistor_to_transistordatabase_file_exchange)






        ### Actions for 'help' ###
        self.action_contribute.triggered.connect(self.webbrowser_contribute)
        self.action_documentation.triggered.connect(self.webbrowser_documentation)
        self.action_report_bug.triggered.connect(self.webbrowser_bugreport)


        ###TOPOLOGY CALCULATOR###

        # create the six Matplotlib figures needed
        self.matplotlibwidget_topology1 = MatplotlibWidget()
        self.matplotlibwidget_topology2 = MatplotlibWidget()
        self.matplotlibwidget_topology3 = MatplotlibWidget()
        self.matplotlibwidget_topology4 = MatplotlibWidget()
        self.matplotlibwidget_topology5 = MatplotlibWidget()
        self.matplotlibwidget_topology6 = MatplotlibWidget()


        # fill all ComboBoxes when starting the program

        items_comboBox_topology_topology = ["Buck-Converter", "Boost-Converter", "Buck-Boost-Converter" ]
        self.comboBox_topology_topology.addItems(items_comboBox_topology_topology)
        self.comboBox_topology_transistor1.addItems(transistor_list)
        self.comboBox_topology_transistor2.addItems(transistor_list)


        items_comboBox_line_contour = ["Contour", "Line"]
        self.comboBox_topology_plot1_line_contour.addItems(items_comboBox_line_contour)
        self.comboBox_topology_plot2_line_contour.addItems(items_comboBox_line_contour)
        self.comboBox_topology_plot3_line_contour.addItems(items_comboBox_line_contour)
        self.comboBox_topology_plot4_line_contour.addItems(items_comboBox_line_contour)
        self.comboBox_topology_plot5_line_contour.addItems(items_comboBox_line_contour)
        self.comboBox_topology_plot6_line_contour.addItems(items_comboBox_line_contour)

        items_comboBox_topology_x_axis = ["Vin [V]", "Vout [V]", "Output Power [W]", "Frequency [kHz]", "Zeta = f*L"]
        items_comboBox_topology_y_axis = ["Vin [V]", "Vout [V]", "Output Power [W]", "Frequency [kHz]", "Zeta = f*L"]
        items_comboBox_topology_z_axis = ["RMS Current Transistor1 [A]",
                                 "RMS Current Diode Transistor2 [A]",
                                 "Mean Current Transistor1 [A]",
                                 "Mean Current Diode Transistor2 [A]",
                                 "RMS Inductor Current [A]",
                                 "Mean Inductor Current [A]",
                                 "Peak Current [A]",
                                 "Conduction Losses Transistor1 [W]",
                                 "Conduction Losses Diode Transistor2 [W]",
                                 "Total Conduction Losses [W]",
                                 "Turn-on Switching Losses Transistor1 [W]",
                                 "Turn-off Switching Losses Transistor1 [W]",
                                 "Reverse Recovery Losses Diode Transistor2 [W]",
                                 "Total Switching Losses Transistor1 [W]",
                                 "Total Power Losses Transistor1 [W]",
                                 "Total Switching Losses [W]",
                                 "Temperature Switch Transistor1 [°C]",
                                 "Temperature Diode Transistor2 [°C]"]

        self.comboBox_topology_plot1_x_axis.addItems(items_comboBox_topology_x_axis)
        self.comboBox_topology_plot1_y_axis.addItems(items_comboBox_topology_y_axis)
        self.comboBox_topology_plot1_z_axis.addItems(items_comboBox_topology_z_axis)

        self.comboBox_topology_plot2_x_axis.addItems(items_comboBox_topology_x_axis)
        self.comboBox_topology_plot2_y_axis.addItems(items_comboBox_topology_y_axis)
        self.comboBox_topology_plot2_z_axis.addItems(items_comboBox_topology_z_axis)

        self.comboBox_topology_plot3_x_axis.addItems(items_comboBox_topology_x_axis)
        self.comboBox_topology_plot3_y_axis.addItems(items_comboBox_topology_y_axis)
        self.comboBox_topology_plot3_z_axis.addItems(items_comboBox_topology_z_axis)

        self.comboBox_topology_plot4_x_axis.addItems(items_comboBox_topology_x_axis)
        self.comboBox_topology_plot4_y_axis.addItems(items_comboBox_topology_y_axis)
        self.comboBox_topology_plot4_z_axis.addItems(items_comboBox_topology_z_axis)


        self.comboBox_topology_plot5_x_axis.addItems(items_comboBox_topology_x_axis)
        self.comboBox_topology_plot5_y_axis.addItems(items_comboBox_topology_y_axis)
        self.comboBox_topology_plot5_z_axis.addItems(items_comboBox_topology_z_axis)

        self.comboBox_topology_plot6_x_axis.addItems(items_comboBox_topology_x_axis)
        self.comboBox_topology_plot6_y_axis.addItems(items_comboBox_topology_y_axis)
        self.comboBox_topology_plot6_z_axis.addItems(items_comboBox_topology_z_axis)


        # connect the update button with the function to update the plots
        self.button_topology_update_plots.clicked.connect(self.topology_update_plots)

        # connect the buttons to pop-out plots with the function to pop-out the plot
        self.button_topology_pop_out_plot1.clicked.connect(self.topology_pop_out_plot1)
        self.button_topology_pop_out_plot2.clicked.connect(self.topology_pop_out_plot2)
        self.button_topology_pop_out_plot3.clicked.connect(self.topology_pop_out_plot3)
        self.button_topology_pop_out_plot4.clicked.connect(self.topology_pop_out_plot4)
        self.button_topology_pop_out_plot5.clicked.connect(self.topology_pop_out_plot5)
        self.button_topology_pop_out_plot6.clicked.connect(self.topology_pop_out_plot6)

        # run the functions "comboBox_topology_transistor1_changed" and "comboBox_topology_transistor2_changed",
        # when the current text of the ComboBoxes to choose the transistors changed, which will reset and refill
        # the ComboBoxes that store transistor specific selections(gate-voltages and gate-resistors)
        
        self.start=True
        self.comboBox_topology_transistor1.currentTextChanged.connect(self.comboBox_topology_transistor1_changed)
        self.comboBox_topology_transistor1_changed()

        # run the function "slider_topology_r_g_value_changed" when a slider to choose the gate resistor changed,
        # which will then change the Text of the Label below to show current selection
        self.slider_topology_r_g_on_transistor1.valueChanged.connect(self.slider_topology_r_g_value_changed)
        self.slider_topology_r_g_off_transistor1.valueChanged.connect(self.slider_topology_r_g_value_changed)

        # run the function "comboBox_topology_topology_changed" when the current text of the ComboBox to choose the
        # topology changed, which will set the picture of the the circuit diagram below for chosen topology
        self.comboBox_topology_topology.currentTextChanged.connect(self.comboBox_topology_topology_changed)
        self.comboBox_topology_topology_changed()

        # run the function "comboBox_topology_plots_line_contour_changed", when the current text of a ComboBox to choose
        # if a plot should be in line or contour
        self.comboBox_topology_plot1_line_contour.currentTextChanged.connect(self.comboBox_topology_plots_line_contour_changed)
        self.comboBox_topology_plot2_line_contour.currentTextChanged.connect(self.comboBox_topology_plots_line_contour_changed)
        self.comboBox_topology_plot3_line_contour.currentTextChanged.connect(self.comboBox_topology_plots_line_contour_changed)
        self.comboBox_topology_plot3_line_contour.currentTextChanged.connect(self.comboBox_topology_plots_line_contour_changed)
        self.comboBox_topology_plot4_line_contour.currentTextChanged.connect(self.comboBox_topology_plots_line_contour_changed)
        self.comboBox_topology_plot5_line_contour.currentTextChanged.connect(self.comboBox_topology_plots_line_contour_changed)
        self.comboBox_topology_plot6_line_contour.currentTextChanged.connect(self.comboBox_topology_plots_line_contour_changed)

        # run the function "comboBox_topology_plots_x_axis_changed", when the current text of a CombBox to choose
        # the variable for the x-axis changed
        self.comboBox_topology_plot1_x_axis.currentTextChanged.connect(self.comboBox_topology_plots_x_axis_changed)
        self.comboBox_topology_plot2_x_axis.currentTextChanged.connect(self.comboBox_topology_plots_x_axis_changed)
        self.comboBox_topology_plot3_x_axis.currentTextChanged.connect(self.comboBox_topology_plots_x_axis_changed)
        self.comboBox_topology_plot3_x_axis.currentTextChanged.connect(self.comboBox_topology_plots_x_axis_changed)
        self.comboBox_topology_plot4_x_axis.currentTextChanged.connect(self.comboBox_topology_plots_x_axis_changed)
        self.comboBox_topology_plot5_x_axis.currentTextChanged.connect(self.comboBox_topology_plots_x_axis_changed)
        self.comboBox_topology_plot6_x_axis.currentTextChanged.connect(self.comboBox_topology_plots_x_axis_changed)

        # Topology Calculator: set input masks and validators
        self.lineEdit_topology_output_power.setValidator(QDoubleValidator())
        self.lineEdit_topology_v_in.setValidator(QDoubleValidator())
        self.lineEdit_topology_v_out.setValidator(QDoubleValidator())
        self.lineEdit_topology_frequency.setValidator(QDoubleValidator())
        self.lineEdit_topology_zeta.setValidator(QDoubleValidator())
        self.lineEdit_topology_temperature_heatsink.setValidator(QDoubleValidator())
        self.lineEdit_topology_thermal_resistance_heatsink.setValidator(QDoubleValidator())
        self.lineEdit_topology_number_parallel_transistor1.setValidator(QDoubleValidator())
        self.lineEdit_topology_output_power_min.setValidator(QDoubleValidator())
        self.lineEdit_topology_v_in_min.setValidator(QDoubleValidator())
        self.lineEdit_topology_v_out_min.setValidator(QDoubleValidator())
        self.lineEdit_topology_frequency_min.setValidator(QDoubleValidator())
        self.lineEdit_topology_zeta_min.setValidator(QDoubleValidator())
        self.lineEdit_topology_number_parallel_transistor2.setValidator(QDoubleValidator())
        self.lineEdit_topology_output_power_max.setValidator(QDoubleValidator())
        self.lineEdit_topology_v_in_max.setValidator(QDoubleValidator())
        self.lineEdit_topology_v_out_max.setValidator(QDoubleValidator())
        self.lineEdit_topology_frequency_max.setValidator(QDoubleValidator())
        self.lineEdit_topology_zeta_max.setValidator(QDoubleValidator())

        ###COMPARISON TOOLS###

        self.comboBox_compare_transistor1.setStyleSheet("color: blue")
        self.comboBox_compare_transistor2.setStyleSheet("color: green")
        self.comboBox_compare_transistor3.setStyleSheet("color: red")

        # fill the comboBoxes to choose transistors
        self.comboBox_compare_transistor1.addItems(transistor_list)
        self.comboBox_compare_transistor2.addItems(transistor_list)
        self.comboBox_compare_transistor3.addItems(transistor_list)


        # connect the button to update the plots with the function to update the plots
        self.button_compare_update_plots.clicked.connect(self.compare_update_plots)

        # connect the buttons to pop-out plots with the function to pop-out the plot
        self.button_compare_pop_out_plot1.clicked.connect(self.compare_pop_out_plot1)
        self.button_compare_pop_out_plot2.clicked.connect(self.compare_pop_out_plot2)
        self.button_compare_pop_out_plot3.clicked.connect(self.compare_pop_out_plot3)
        self.button_compare_pop_out_plot4.clicked.connect(self.compare_pop_out_plot4)
        self.button_compare_pop_out_plot5.clicked.connect(self.compare_pop_out_plot5)
        self.button_compare_pop_out_plot6.clicked.connect(self.compare_pop_out_plot6)
        self.button_compare_pop_out_plot7.clicked.connect(self.compare_pop_out_plot7)
        self.button_compare_pop_out_plot8.clicked.connect(self.compare_pop_out_plot8)
        self.button_compare_pop_out_plot9.clicked.connect(self.compare_pop_out_plot9)

        # fill the comboBoxes to select the data to plot
        items_comboBoxes_compare = ["Switch Channel Data",
                                    "Diode Channel Data",
                                    "Switch Turn-on Losses",
                                    "Switch Turn-off Losses",
                                    "Diode Reverse Recovery Losses",
                                    "Output Capacitance Charge vs. Channel Voltage",
                                    "Output Capacitance Energy vs. Channel Voltage",
                                    "Switch Energy Data vs. Channel Current Transistor1",
                                    "Diode Energy Data vs. Channel Current Transistor1",
                                    "Switch Energy Data vs. Gate Resistor Transistor1",
                                    "Diode Energy Data vs. Gate Resistor Transistor1",
                                    "Switch Energy Data vs. Channel Current Transistor2",
                                    "Diode Energy Data vs. Channel Current Transistor2",
                                    "Switch Energy Data vs. Gate Resistor Transistor2",
                                    "Diode Energy Data vs. Gate Resistor Transistor2",
                                    "Switch Energy Data vs. Channel Current Transistor3",
                                    "Diode Energy Data vs. Channel Current Transistor3",
                                    "Switch Energy Data vs. Gate Resistor Transistor3",
                                    "Diode Energy Data vs. Gate Resistor Transistor3",
                                    "Switch Channel Data Transistor1",
                                    "Diode Channel Data Transistor1",
                                    "Switch Channel Data Transistor2",
                                    "Diode Channel Data Transistor2",
                                    "Switch Channel Data Transistor3",
                                    "Diode Channel Data Transistor3"]

        self.comboBox_compare_plot1.addItems(items_comboBoxes_compare)
        self.comboBox_compare_plot2.addItems(items_comboBoxes_compare)
        self.comboBox_compare_plot3.addItems(items_comboBoxes_compare)
        self.comboBox_compare_plot4.addItems(items_comboBoxes_compare)
        self.comboBox_compare_plot5.addItems(items_comboBoxes_compare)
        self.comboBox_compare_plot6.addItems(items_comboBoxes_compare)
        self.comboBox_compare_plot7.addItems(items_comboBoxes_compare)
        self.comboBox_compare_plot8.addItems(items_comboBoxes_compare)
        self.comboBox_compare_plot9.addItems(items_comboBoxes_compare)

        # Create the nine Matplotlib figures needed
        self.matplotlibwidget_compare1 = MatplotlibWidget()
        self.matplotlibwidget_compare2 = MatplotlibWidget()
        self.matplotlibwidget_compare3 = MatplotlibWidget()
        self.matplotlibwidget_compare4 = MatplotlibWidget()
        self.matplotlibwidget_compare5 = MatplotlibWidget()
        self.matplotlibwidget_compare6 = MatplotlibWidget()
        self.matplotlibwidget_compare7 = MatplotlibWidget()
        self.matplotlibwidget_compare8 = MatplotlibWidget()
        self.matplotlibwidget_compare9 = MatplotlibWidget()

        # run the function "slider_topology_r_g_value_changed" when a slider to choose the gate resistor changed,
        # which will then change the Text of the Label below to show current selection
        self.slider_compare_r_g_on_transistor1.valueChanged.connect(self.slider_compare_r_g_value_changed)
        self.slider_compare_r_g_off_transistor1.valueChanged.connect(self.slider_compare_r_g_value_changed)
        self.slider_compare_r_g_on_transistor2.valueChanged.connect(self.slider_compare_r_g_value_changed)
        self.slider_compare_r_g_off_transistor2.valueChanged.connect(self.slider_compare_r_g_value_changed)
        self.slider_compare_r_g_on_transistor3.valueChanged.connect(self.slider_compare_r_g_value_changed)
        self.slider_compare_r_g_off_transistor3.valueChanged.connect(self.slider_compare_r_g_value_changed)

        # run the function "comboBox_compare_transistors_changed", when the current text of a comboBox to choose the
        # transistors changed
        self.comboBox_compare_transistor1.currentTextChanged.connect(self.comboBox_compare_transistor1_changed)
        self.comboBox_compare_transistor2.currentTextChanged.connect(self.comboBox_compare_transistor2_changed)
        self.comboBox_compare_transistor3.currentTextChanged.connect(self.comboBox_compare_transistor3_changed)
        self.comboBox_compare_transistor1_changed()
        self.comboBox_compare_transistor2_changed()
        self.comboBox_compare_transistor3_changed()

        # Comparison Tools: set input masks and validators
        self.lineEdit_compare_t_j_transistor1.setValidator(QDoubleValidator())
        self.lineEdit_compare_t_j_transistor2.setValidator(QDoubleValidator())
        self.lineEdit_compare_t_j_transistor3.setValidator(QDoubleValidator())
        self.lineEdit_compare_v_supply_transistor1.setValidator(QDoubleValidator())
        self.lineEdit_compare_v_supply_transistor2.setValidator(QDoubleValidator())
        self.lineEdit_compare_v_supply_transistor3.setValidator(QDoubleValidator())
        self.lineEdit_compare_number_parallel_transistor1.setValidator(QDoubleValidator())
        self.lineEdit_compare_number_parallel_transistor2.setValidator(QDoubleValidator())
        self.lineEdit_compare_number_parallel_transistor3.setValidator(QDoubleValidator())
        
        ###Exporting Tools###

        # add Items to the comboBox and set standard values for parallel transistor and normalize-t-to-v value
        self.comboBox_export_transistor.addItems(transistor_list)

        # connect the buttons to the functions
        self.button_export_datasheet.clicked.connect(self.export_datasheet)
        self.button_export_json.clicked.connect(self.export_json)
        self.button_export_matlab.clicked.connect(self.export_matlab)
        self.button_export_simulink.clicked.connect(self.export_simulink)
        self.button_export_gecko.clicked.connect(self.export_gecko)
        self.button_export_plecs.clicked.connect(self.export_plecs)

        # Exporting Tools: set input masks and validators for transistor parameters
        self.lineEdit_export_number_parallel_transistors.setValidator(QDoubleValidator())
        self.lineEdit_export_simulink_v_supply.setValidator(QDoubleValidator())
        self.lineEdit_export_simulink_r_g_on.setValidator(QDoubleValidator())
        self.lineEdit_export_simulink_r_g_off.setValidator(QDoubleValidator())
        self.lineEdit_export_simulink_normalize_t_to_v.setValidator(QDoubleValidator())
        self.lineEdit_export_gecko_v_supply.setValidator(QDoubleValidator())
        self.lineEdit_export_gecko_r_g_on.setValidator(QDoubleValidator())
        self.lineEdit_export_gecko_r_g_off.setValidator(QDoubleValidator())
        self.lineEdit_export_gecko_v_g_on.setValidator(QDoubleValidator())
        self.lineEdit_export_gecko_v_g_off.setValidator(QDoubleValidator())
        self.lineEdit_export_plecs_v_g_on.setValidator(QDoubleValidator())
        self.lineEdit_export_plecs_v_g_off.setValidator(QDoubleValidator())
        self.lineEdit_export_plecs_v_d_on.setValidator(QDoubleValidator())
        self.lineEdit_export_plecs_v_d_off.setValidator(QDoubleValidator())


        ###CREATE TRANSISTOR###

        self.fill_comboBoxes_create_transistor()

        for widget in self.scrollAreaWidgetContents_create_transistor.children():
            if isinstance(widget, QComboBox) or isinstance(widget, QLineEdit):
                widget.setStyleSheet("color: green")

        # connect all buttons to their specific functions
        self.button_create_transistor_switch_add_curve_channel_data.clicked.connect(self.add_curve_switch_channel)
        self.button_create_transistor_switch_delete_curve_channel_data.clicked.connect(self.delete_curve_switch_channel)

        self.button_create_transistor_switch_add_curve_switching_losses.clicked.connect(
            self.add_curve_switch_switching_losses)
        self.button_create_transistor_switch_delete_curve_switching_losses.clicked.connect(
            self.delete_curve_switch_switching_losses)

        self.button_create_transistor_switch_add_curve_gate_charge.clicked.connect(self.add_curve_switch_gate_charge)
        self.button_create_transistor_switch_delete_curve_gate_charge.clicked.connect(
            self.delete_curve_switch_gate_charge)

        self.button_create_transistor_switch_add_curve_r_on.clicked.connect(self.add_curve_switch_r_on)
        self.button_create_transistor_switch_delete_curve_r_on.clicked.connect(self.delete_curve_switch_r_on)

        self.button_create_transistor_diode_add_curve_channel_data.clicked.connect(self.add_curve_diode_channel)
        self.button_create_transistor_diode_delete_curve_channel_data.clicked.connect(self.delete_curve_diode_channel)

        self.button_create_transistor_diode_add_curve_switching_losses.clicked.connect(
            self.add_curve_diode_switching_losses)
        self.button_create_transistor_diode_delete_curve_switching_losses.clicked.connect(
            self.delete_curve_diode_switching_losses)

        self.button_create_transistor_switch_add_t_rthjc_curve.clicked.connect(self.add_curve_switch_t_rthjc)
        self.button_create_transistor_switch_delete_curve_t_rthjc.clicked.connect(self.delete_curve_switch_t_rthjc)

        self.button_create_transistor_diode_add_t_rthjc_curve.clicked.connect(self.add_curve_diode_t_rthjc)
        self.button_create_transistor_diode_delete_curve_t_rthjc.clicked.connect(self.delete_curve_diode_t_rthjc)

        self.button_create_transistor_add_curve_v_ecoss.clicked.connect(self.add_curve_v_ecoss)
        self.button_create_transistor_delete_curve_v_ecoss.clicked.connect(self.delete_curve_v_ecoss)

        self.button_create_transistor_add_curve_c_iss_normal.clicked.connect(self.add_curve_c_iss_normal)
        self.button_create_transistor_delete_curve_c_iss_normal.clicked.connect(self.delete_curve_c_iss_normal)

        self.button_create_transistor_add_curve_c_iss_detail.clicked.connect(self.add_curve_c_iss_detail)
        self.button_create_transistor_delete_curve_c_iss_detail.clicked.connect(self.delete_curve_c_iss_detail)

        self.button_create_transistor_add_curve_c_oss_normal.clicked.connect(self.add_curve_c_oss_normal)
        self.button_create_transistor_delete_curve_c_oss_normal.clicked.connect(self.delete_curve_c_oss_normal)

        self.button_create_transistor_add_curve_c_oss_detail.clicked.connect(self.add_curve_c_oss_detail)
        self.button_create_transistor_delete_curve_c_oss_detail.clicked.connect(self.delete_curve_c_oss_detail)

        self.button_create_transistor_add_curve_c_rss_normal.clicked.connect(self.add_curve_c_rss_normal)
        self.button_create_transistor_delete_curve_c_rss_normal.clicked.connect(self.delete_curve_c_rss_normal)

        self.button_create_transistor_add_curve_c_rss_detail.clicked.connect(self.add_curve_c_rss_detail)
        self.button_create_transistor_delete_curve_c_rss_detail.clicked.connect(self.delete_curve_c_rss_detail)

        self.button_create_transistor_switch_add_curve_soa_t_pulse.clicked.connect(self.add_curve_switch_soa_t_pulse)
        self.button_create_transistor_switch_delete_curve_soa_t_pulse.clicked.connect(self.delete_curve_switch_soa_t_pulse)

        self.button_create_transistor_diode_add_curve_soa_t_pulse.clicked.connect(self.add_curve_diode_soa_t_pulse)
        self.button_create_transistor_diode_delete_curve_soa_t_pulse.clicked.connect(self.delete_curve_diode_soa_t_pulse)

        self.button_create_transistor_create.clicked.connect(self.load_transistor_into_local_database)
        self.button_create_transistor_preview_transistor.clicked.connect(self.preview_transistor_on_virtual_datasheet)

        self.button_create_transistor_view_curve_switch_channel_data.clicked.connect(self.view_curve_switch_channel)
        self.button_create_transistor_view_curve_switch_switching_losses.clicked.connect(self.view_curve_switch_switching_losses)
        self.button_create_transistor_view_curve_switch_gate_charge.clicked.connect(self.view_curve_switch_gate_charge)
        self.button_create_transistor_view_curve_switch_r_on.clicked.connect(self.view_curve_switch_r_on)
        self.button_create_transistor_view_curve_switch_soa_t_pulse.clicked.connect(self.view_curve_switch_soa_t_pulse)
        self.button_create_transistor_view_curve_diode_channel_data.clicked.connect(self.view_curve_diode_channel)
        self.button_create_transistor_view_curve_diode_switching_losses.clicked.connect(self.view_curve_diode_switching_losses)
        self.button_create_transistor_view_curve_diode_soa_t_pulse.clicked.connect(self.view_curve_diode_soa_t_pulse)
        self.button_create_transistor_view_curve_switch_t_rthjc.clicked.connect(self.view_curve_switch_t_rthjc)
        self.button_create_transistor_view_curve_diode_t_rthjc.clicked.connect(self.view_curve_diode_t_rthjc)
        self.button_create_transistor_view_curve_v_ecoss.clicked.connect(self.view_curve_v_ecoss)
        self.button_create_transistor_view_curve_c_iss_normal.clicked.connect(self.view_curve_c_iss_normal)
        self.button_create_transistor_view_curve_c_iss_detail.clicked.connect(self.view_curve_c_iss_detail)
        self.button_create_transistor_view_curve_c_oss_normal.clicked.connect(self.view_curve_c_oss_normal)
        self.button_create_transistor_view_curve_c_oss_detail.clicked.connect(self.view_curve_c_oss_detail)
        self.button_create_transistor_view_curve_c_rss_normal.clicked.connect(self.view_curve_c_rss_normal)
        self.button_create_transistor_view_curve_c_rss_detail.clicked.connect(self.view_curve_c_rss_detail)

        self.button_create_transistor_add_directory_dpt.clicked.connect(self.add_dpt_measurement_data_directory)
        self.button_create_transistor_view_directory_dpt.clicked.connect(self.view_dpt_measurement_data)
        self.button_create_transistor_delete_directory_dpt.clicked.connect(self.delete_dpt_measurement_data)


        # set mandatory example keys
        # self.lineEdit_create_transistor_name.setText("Test")
        # self.lineEdit_create_transistor_author.setText("Phillip Beine")
        # self.lineEdit_create_transistor_cooling_area.setText("160e-6")
        # self.lineEdit_create_transistor_housing_area.setText("367e-6")
        # self.lineEdit_create_transistor_r_th_diode_cs.setText("0")
        # self.lineEdit_create_transistor_r_th_switch_cs.setText("0")
        # self.lineEdit_create_transistor_i_abs_max.setText("250")
        # self.lineEdit_create_transistor_v_abs_max.setText("1200")
        # self.lineEdit_create_transistor_i_cont.setText("115")
        # self.lineEdit_create_transistor_r_g_int.setText("2.6")
        # self.lineEdit_create_transistor_r_th_cs.setText("0")
        # self.lineEdit_create_transistor_switch_r_th_total.setText("0.27")
        # self.lineEdit_create_transistor_diode_r_th_total.setText("0")
        # self.lineEdit_create_transistor_switch_t_j_max.setText("175")
        # self.lineEdit_create_transistor_diode_t_j_max.setText("175")

        self.lineEdit_create_transistor_add_data_dpt_measurement_date.setText(str(datetime.date.today()))

        # Testbench combobox
        self.comboBox_create_transistor_add_data_dpt_measurement_testbench.addItems(['', 'Paderborn University / LEA'])
        self.comboBox_create_transistor_add_data_dpt_measurement_testbench.setEditable(True)


        #Sets all settings given in the settings.json file in the current working directory
        self.load_local_settings()


        # create transistor: set input masks and validators for transistor parameters
        self.lineEdit_create_transistor_v_abs_max.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_i_abs_max.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_i_cont.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_transistor_c_oss_er_c_o.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_transistor_c_oss_er_v_gs.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_transistor_c_oss_er_v_ds.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_transistor_c_oss_tr_c_o.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_transistor_c_oss_tr_v_gs.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_transistor_c_oss_tr_v_ds.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_t_c_max.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_r_g_int.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_r_th_cs.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_r_th_diode_cs.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_r_th_switch_cs.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_r_g_on_recommended.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_r_g_off_recommended.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_c_iss_fix.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_c_oss_fix.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_c_rss_fix.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_add_curve_c_iss_t_j.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_add_curve_c_oss_t_j.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_add_curve_c_rss_t_j.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_add_data_dpt_load_inductance.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_add_data_commutation_inductance.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_add_data_dpt_v_g.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_add_data_dpt_v_g_off.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_add_data_dpt_r_g_on.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_add_data_dpt_r_g_off.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_add_data_dpt_t_j.setValidator(QDoubleValidator())

        # set input masks and validators for switch parameters
        self.lineEdit_create_transistor_switch_t_j_max.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_switch_r_th_total.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_switch_c_th_total.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_switch_tau_total.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_switch_add_channel_data_t_j.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_switch_add_channel_data_v_g.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_switch_add_switching_losses_t_j.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_switch_add_switching_losses_v_g.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_switch_add_switching_losses_r_g.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_switch_add_switching_losses_v_supply.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_switch_add_gate_charge_i_channel.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_switch_add_gate_charge_t_j.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_switch_add_gate_charge_v_supply.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_switch_add_gate_charge_i_g.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_switch_add_r_on_i_channel.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_switch_add_r_on_v_g.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_switch_add_r_on_r_channel_nominal.setValidator(QDoubleValidator())

        self.lineEdit_create_transistor_switch_soa_t_pulse_time_pulse.setValidator(QDoubleValidator(0.0, 1000.0, 8))
        self.lineEdit_create_transistor_switch_soa_t_pulse_t_c.setValidator(QDoubleValidator(-300.0, 300.0, 2))

        # set input masks and validators for diode parameters
        self.lineEdit_create_transistor_diode_t_j_max.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_diode_r_th_total.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_diode_c_th_total.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_diode_tau_total.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_diode_add_channel_data_t_j.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_diode_add_channel_data_v_g.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_diode_add_switching_losses_t_j.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_diode_add_switching_losses_v_g.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_diode_add_switching_losses_r_g.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_diode_add_switching_losses_v_supply.setValidator(QDoubleValidator())
        self.lineEdit_create_transistor_diode_soa_t_pulse_time_pulse.setValidator(QDoubleValidator(0.0, 1000.0, 8))
        self.lineEdit_create_transistor_diode_soa_t_pulse_t_c.setValidator(QDoubleValidator(-300, 300, 2))

        ###SEARCH DATABASE###

        # fill the comboBoxes
        self.comboBox_search_database_load_comparison_tools.addItems(["Transistor1", "Transistor2", "Transistor3"])
        self.comboBox_search_database_load_topology_calculator.addItems(["Transistor1", "Transistor2"])

        # connect the buttons to the functions
        self.button_search_database_load_exporting_tools.clicked.connect(self.load_from_search_database_into_exporting_tools)
        self.button_search_database_load_comparison_tools.clicked.connect(self.load_from_search_database_into_comparison_tools)
        self.button_search_database_load_topology_calculator.clicked.connect(self.load_from_search_database_into_topology_calculator)
        self.button_search_database_load_create_transistor.clicked.connect(self.load_from_search_database_into_create_transistor)
        self.button_search_database_reset_filters.clicked.connect(self.reset_filter_search_database)
        self.button_update_from_fileexchange.clicked.connect(self.update_database_from_fileexchange)

        # connect all checkBoxes(when state changed) and lineEdits(when text changed) to the functions to load data into the tableWidget
        self.checkBox_search_database_id.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_name.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_type.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_author.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_technology.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_template_version.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_template_date.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_creation_date.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_last_modified.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_comment.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_datasheet_hyperlink.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_datasheet_date.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_datasheet_version.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_housing_area.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_cooling_area.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_r_g_int.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_t_c_max.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_r_g_on_recommended.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_r_g_off_recommended.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_c_oss_fix.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_c_iss_fix.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_c_rss_fix.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_housing_type.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_manufacturer.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_r_th_cs.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_r_th_switch_cs.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_r_th_diode_cs.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_v_abs_max.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_i_abs_max.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_i_cont.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_switch_t_j_max.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_switch_comment.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_switch_manufacturer.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_switch_technology.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_diode_comment.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_diode_manufacturer.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_diode_technology.stateChanged.connect(self.search_database_load_data)
        self.checkBox_search_database_diode_t_j_max.stateChanged.connect(self.search_database_load_data)

        self.lineEdit_search_database_housing_area_min.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_cooling_area_min.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_r_g_int_min.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_t_c_max_min.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_r_g_on_recommended_min.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_r_g_off_recommended_min.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_c_oss_fix_min.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_c_iss_fix_min.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_c_rss_fix_min.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_r_th_cs_min.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_r_th_switch_cs_min.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_r_th_diode_cs_min.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_v_abs_max_min.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_i_abs_max_min.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_i_cont_min.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_switch_t_j_max_min.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_diode_t_j_max_min.textChanged.connect(self.search_database_load_data)

        self.lineEdit_search_database_housing_area_max.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_cooling_area_max.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_r_g_int_max.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_t_c_max_max.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_r_g_on_recommended_max.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_r_g_off_recommended_max.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_c_oss_fix_max.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_c_iss_fix_max.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_c_rss_fix_max.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_r_th_cs_max.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_r_th_switch_cs_max.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_r_th_diode_cs_max.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_v_abs_max_max.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_i_abs_max_max.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_i_cont_max.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_switch_t_j_max_max.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_diode_t_j_max_max.textChanged.connect(self.search_database_load_data)

        self.lineEdit_search_database_id.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_name.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_type.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_author.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_technology.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_template_version.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_template_date.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_creation_date.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_last_modified.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_comment.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_datasheet_hyperlink.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_datasheet_date.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_datasheet_version.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_housing_type.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_manufacturer.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_switch_comment.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_switch_manufacturer.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_switch_technology.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_diode_comment.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_diode_manufacturer.textChanged.connect(self.search_database_load_data)
        self.lineEdit_search_database_diode_technology.textChanged.connect(self.search_database_load_data)

        # load data into the tableWidget when starting the program
        self.search_database_load_data()

        # Search Database: set input masks and validators
        self.lineEdit_search_database_housing_area_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_housing_area_max.setValidator(QDoubleValidator())
        self.lineEdit_search_database_cooling_area_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_cooling_area_max.setValidator(QDoubleValidator())
        self.lineEdit_search_database_t_c_max_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_t_c_max_max.setValidator(QDoubleValidator())
        self.lineEdit_search_database_r_g_int_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_r_g_int_max.setValidator(QDoubleValidator())
        self.lineEdit_search_database_r_g_on_recommended_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_r_g_on_recommended_max.setValidator(QDoubleValidator())
        self.lineEdit_search_database_r_g_off_recommended_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_r_g_off_recommended_max.setValidator(QDoubleValidator())
        self.lineEdit_search_database_c_oss_fix_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_c_oss_fix_max.setValidator(QDoubleValidator())
        self.lineEdit_search_database_c_iss_fix_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_c_iss_fix_max.setValidator(QDoubleValidator())
        self.lineEdit_search_database_c_rss_fix_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_c_rss_fix_max.setValidator(QDoubleValidator())
        self.lineEdit_search_database_r_th_cs_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_r_th_cs_max.setValidator(QDoubleValidator())
        self.lineEdit_search_database_r_th_switch_cs_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_r_th_switch_cs_max.setValidator(QDoubleValidator())
        self.lineEdit_search_database_r_th_diode_cs_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_r_th_diode_cs_max.setValidator(QDoubleValidator())
        self.lineEdit_search_database_v_abs_max_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_v_abs_max_max.setValidator(QDoubleValidator())
        self.lineEdit_search_database_i_abs_max_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_i_abs_max_max.setValidator(QDoubleValidator())
        self.lineEdit_search_database_i_cont_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_i_cont_max.setValidator(QDoubleValidator())
        self.lineEdit_search_database_switch_t_j_max_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_switch_t_j_max_max.setValidator(QDoubleValidator())
        self.lineEdit_search_database_diode_t_j_max_min.setValidator(QDoubleValidator())
        self.lineEdit_search_database_diode_t_j_max_max.setValidator(QDoubleValidator())

    ### Transistor actions ###
    def webbrowser_original_datasheet(self):
        transistor = self.get_marked_transistor()
        webbrowser.open(transistor.datasheet_hyperlink)

    def webbrowser_virtual_datasheet(self):
        transistor = self.get_marked_transistor()
        html = transistor.export_datasheet(build_collection=True)
        with tempfile.NamedTemporaryFile('w', delete=False, suffix='.html') as f:
            url = 'file://' + f.name
            f.write(html)
        webbrowser.open(url)

    def email_add_transistor_to_transistordatabase_file_exchange(self):
        """
        Email workflow to start a request for adding a new transistor to the transistordatabase file exchange.
        This routine will open the mailprogram with predefined adresses. The .json file needs to be added manually.
        """

        transistor = self.get_marked_transistor()
        transistor.export_json()

        self.show_popup_message(f'Workflow to start request for upload <b>{transistor.name}</b> to the transistordatabase file exchange: <br> <br> 1. The browser opens and wants to access the mail program. Allow this.  <br> 2. The email program opens with the addressee pre-filled. <br> 3. Add the transistor file {transistor.name}.json as attachment from this filder: <a href={pathlib.Path.cwd().as_uri()}>{pathlib.Path.cwd().as_uri()}</a> <br> 4. Send Email.')

        email_body = f"Do not forget to attach the transistor file <b>{transistor.name}</b> to this email!! Link to File: <a href={pathlib.Path.cwd().as_uri()}>{pathlib.Path.cwd().as_uri()}</a>"
        email_subject = 'Request to add transistor {transistor.name} to the transistordatabase file exchange (TDB-FE)'
        webbrowser.open(f'mailto:?to=tdb@lea.upb.de&subject=' + email_subject + '&body=' + email_body, new=2)


    ### Help actions ###
    def webbrowser_contribute(self):
        webbrowser.open('https://github.com/upb-lea/transistordatabase/blob/main/Contributing.rst')

    def webbrowser_bugreport(self):
        webbrowser.open('https://github.com/upb-lea/transistordatabase/issues')

    def webbrowser_documentation(self):
        webbrowser.open('https://upb-lea.github.io/transistordatabase/main/transistordatabase.html')




    def closeEvent(self, event):
        """
        Runs when window is closed and saves window size

        :param event: event signal when window is closed
        :return:
        """
        size = self.size()
        self.resize(size)
        self.hide()
        self.move(1,2)
        self.close()


    def get_transistor_list(self):
        """
        Gets list of all transistors stored in local database and returns it

        :return: list with names of all transistors in local database
        """

        mongodb_collection = tdb.connect_local_tdb()
        returned_cursor = mongodb_collection.find()
        name_list = []
        for tran in returned_cursor:
            name_list.append(tran['name'])

        return name_list

    def update_database_from_fileexchange(self):
        """
        Update the local database from the github fileexchange

        :return: None
        :rtype: None
        """
        tdb.update_from_fileexchange()
        self.label_updated_database.setText(f"Successfully Updated")


    def __del__(self):
        """
        Prints standard output in stdout-textEdit

        :return: None
        """
        sys.stdout = sys.__stdout__

    def standard_output_written(self, text):
        cursor = self.textEdit_stdout.textCursor()
        cursor.movePosition(QtGui.QTextCursor.End)
        cursor.insertText(text)
        self.textEdit_stdout.setTextCursor(cursor)
        self.textEdit_stdout.ensureCursorVisible()

    def show_stdout(self):
        """
        Shows stdout-textEdit if the action in the menu bar is checked

        :return: None
        """
        if self.action_stdout.isChecked() == True:
            self.textEdit_stdout.setMaximumSize(16777215, 100)
        else:
            self.textEdit_stdout.setMaximumSize(16777215, 0)


    def get_settings_dict(self):
        """
        Creates and returns a dict containing user inputs and settings

        :return: all_settings_dict: dict containing the name of each widget and its value
        """
        search_database_settings_dict = {"checkBox_search_database_id": self.checkBox_search_database_id.isChecked(),
                                         "checkBox_search_database_name": self.checkBox_search_database_name.isChecked(),
                                         "checkBox_search_database_type": self.checkBox_search_database_type.isChecked(),
                                         "checkBox_search_database_author": self.checkBox_search_database_author.isChecked(),
                                         "checkBox_search_database_technology": self.checkBox_search_database_technology.isChecked(),
                                         "checkBox_search_database_template_version": self.checkBox_search_database_template_version.isChecked(),
                                         "checkBox_search_database_template_date": self.checkBox_search_database_template_date.isChecked(),
                                         "checkBox_search_database_creation_date": self.checkBox_search_database_creation_date.isChecked(),
                                         "checkBox_search_database_last_modified": self.checkBox_search_database_last_modified.isChecked(),
                                         "checkBox_search_database_comment": self.checkBox_search_database_comment.isChecked(),
                                         "checkBox_search_database_datasheet_hyperlink": self.checkBox_search_database_datasheet_hyperlink.isChecked(),
                                         "checkBox_search_database_datasheet_date": self.checkBox_search_database_datasheet_date.isChecked(),
                                         "checkBox_search_database_datasheet_version": self.checkBox_search_database_datasheet_version.isChecked(),
                                         "checkBox_search_database_manufacturer": self.checkBox_search_database_manufacturer.isChecked(),
                                         "checkBox_search_database_housing_type": self.checkBox_search_database_housing_type.isChecked(),
                                         "checkBox_search_database_housing_area": self.checkBox_search_database_housing_area.isChecked(),
                                         "checkBox_search_database_cooling_area": self.checkBox_search_database_cooling_area.isChecked(),
                                         "checkBox_search_database_t_c_max": self.checkBox_search_database_t_c_max.isChecked(),
                                         "checkBox_search_database_r_g_int": self.checkBox_search_database_r_g_int.isChecked(),
                                         "checkBox_search_database_r_g_on_recommended": self.checkBox_search_database_r_g_on_recommended.isChecked(),
                                         "checkBox_search_database_r_g_off_recommended": self.checkBox_search_database_r_g_off_recommended.isChecked(),
                                         "checkBox_search_database_c_oss_fix": self.checkBox_search_database_c_oss_fix.isChecked(),
                                         "checkBox_search_database_c_iss_fix": self.checkBox_search_database_c_iss_fix.isChecked(),
                                         "checkBox_search_database_c_rss_fix": self.checkBox_search_database_c_rss_fix.isChecked(),
                                         "checkBox_search_database_r_th_cs": self.checkBox_search_database_r_th_cs.isChecked(),
                                         "checkBox_search_database_r_th_switch_cs": self.checkBox_search_database_r_th_switch_cs.isChecked(),
                                         "checkBox_search_database_r_th_diode_cs": self.checkBox_search_database_r_th_diode_cs.isChecked(),
                                         "checkBox_search_database_v_abs_max": self.checkBox_search_database_v_abs_max.isChecked(),
                                         "checkBox_search_database_i_abs_max": self.checkBox_search_database_i_abs_max.isChecked(),
                                         "checkBox_search_database_i_cont": self.checkBox_search_database_i_cont.isChecked(),
                                         "checkBox_search_database_switch_comment": self.checkBox_search_database_switch_comment.isChecked(),
                                         "checkBox_search_database_switch_manufacturer": self.checkBox_search_database_switch_manufacturer.isChecked(),
                                         "checkBox_search_database_switch_technology": self.checkBox_search_database_switch_technology.isChecked(),
                                         "checkBox_search_database_switch_t_j_max": self.checkBox_search_database_switch_t_j_max.isChecked(),
                                         "checkBox_search_database_diode_comment": self.checkBox_search_database_diode_comment.isChecked(),
                                         "checkBox_search_database_diode_manufacturer": self.checkBox_search_database_diode_manufacturer.isChecked(),
                                         "checkBox_search_database_diode_technology": self.checkBox_search_database_diode_technology.isChecked(),
                                         "checkBox_search_database_diode_t_j_max": self.checkBox_search_database_diode_t_j_max.isChecked(),
                                         "lineEdit_search_database_id": self.lineEdit_search_database_id.text(),
                                         "lineEdit_search_database_name": self.lineEdit_search_database_name.text(),
                                         "lineEdit_search_database_type": self.lineEdit_search_database_type.text(),
                                         "lineEdit_search_database_author": self.lineEdit_search_database_author.text(),
                                         "lineEdit_search_database_technology": self.lineEdit_search_database_technology.text(),
                                         "lineEdit_search_database_template_version": self.lineEdit_search_database_template_version.text(),
                                         "lineEdit_search_database_template_date": self.lineEdit_search_database_template_date.text(),
                                         "lineEdit_search_database_creation_date": self.lineEdit_search_database_creation_date.text(),
                                         "lineEdit_search_database_last_modified": self.lineEdit_search_database_last_modified.text(),
                                         "lineEdit_search_database_comment": self.lineEdit_search_database_comment.text(),
                                         "lineEdit_search_database_datasheet_hyperlink": self.lineEdit_search_database_datasheet_hyperlink.text(),
                                         "lineEdit_search_database_datasheet_date": self.lineEdit_search_database_datasheet_date.text(),
                                         "lineEdit_search_database_datasheet_version": self.lineEdit_search_database_datasheet_version.text(),
                                         "lineEdit_search_database_manufacturer": self.lineEdit_search_database_manufacturer.text(),
                                         "lineEdit_search_database_housing_type": self.lineEdit_search_database_housing_type.text(),
                                         "lineEdit_search_database_switch_comment": self.lineEdit_search_database_switch_comment.text(),
                                         "lineEdit_search_database_switch_manufacturer": self.lineEdit_search_database_switch_manufacturer.text(),
                                         "lineEdit_search_database_switch_technology": self.lineEdit_search_database_switch_technology.text(),
                                         "lineEdit_search_database_diode_comment": self.lineEdit_search_database_diode_comment.text(),
                                         "lineEdit_search_database_diode_manufacturer": self.lineEdit_search_database_diode_manufacturer.text(),
                                         "lineEdit_search_database_housing_area_min": self.lineEdit_search_database_housing_area_min.text(),
                                         "lineEdit_search_database_housing_area_max": self.lineEdit_search_database_housing_area_max.text(),
                                         "lineEdit_search_database_cooling_area_min": self.lineEdit_search_database_cooling_area_min.text(),
                                         "lineEdit_search_database_cooling_area_max": self.lineEdit_search_database_cooling_area_max.text(),
                                         "lineEdit_search_database_t_c_max_min": self.lineEdit_search_database_t_c_max_min.text(),
                                         "lineEdit_search_database_t_c_max_max": self.lineEdit_search_database_t_c_max_max.text(),
                                         "lineEdit_search_database_r_g_int_min": self.lineEdit_search_database_r_g_int_min.text(),
                                         "lineEdit_search_database_r_g_int_max": self.lineEdit_search_database_r_g_int_max.text(),
                                         "lineEdit_search_database_r_g_on_recommended_min": self.lineEdit_search_database_r_g_on_recommended_min.text(),
                                         "lineEdit_search_database_r_g_on_recommended_max": self.lineEdit_search_database_r_g_on_recommended_max.text(),
                                         "lineEdit_search_database_r_g_off_recommended_min": self.lineEdit_search_database_r_g_off_recommended_min.text(),
                                         "lineEdit_search_database_r_g_off_recommended_max": self.lineEdit_search_database_r_g_off_recommended_max.text(),
                                         "lineEdit_search_database_c_oss_fix_min": self.lineEdit_search_database_c_oss_fix_min.text(),
                                         "lineEdit_search_database_c_oss_fix_max": self.lineEdit_search_database_c_oss_fix_max.text(),
                                         "lineEdit_search_database_c_iss_fix_min": self.lineEdit_search_database_c_iss_fix_min.text(),
                                         "lineEdit_search_database_c_iss_fix_max": self.lineEdit_search_database_c_iss_fix_max.text(),
                                         "lineEdit_search_database_c_rss_fix_min": self.lineEdit_search_database_c_rss_fix_min.text(),
                                         "lineEdit_search_database_c_rss_fix_max": self.lineEdit_search_database_c_rss_fix_max.text(),
                                         "lineEdit_search_database_r_th_cs_min": self.lineEdit_search_database_r_th_cs_min.text(),
                                         "lineEdit_search_database_r_th_cs_max": self.lineEdit_search_database_r_th_cs_max.text(),
                                         "lineEdit_search_database_r_th_switch_cs_min": self.lineEdit_search_database_r_th_switch_cs_min.text(),
                                         "lineEdit_search_database_r_th_switch_cs_max": self.lineEdit_search_database_r_th_switch_cs_max.text(),
                                         "lineEdit_search_database_r_th_diode_cs_min": self.lineEdit_search_database_r_th_diode_cs_min.text(),
                                         "lineEdit_search_database_r_th_diode_cs_max": self.lineEdit_search_database_r_th_diode_cs_max.text(),
                                         "lineEdit_search_database_v_abs_max_min": self.lineEdit_search_database_v_abs_max_min.text(),
                                         "lineEdit_search_database_v_abs_max_max": self.lineEdit_search_database_v_abs_max_max.text(),
                                         "lineEdit_search_database_i_abs_max_min": self.lineEdit_search_database_i_abs_max_min.text(),
                                         "lineEdit_search_database_i_abs_max_max": self.lineEdit_search_database_i_abs_max_max.text(),
                                         "lineEdit_search_database_i_cont_min": self.lineEdit_search_database_i_cont_min.text(),
                                         "lineEdit_search_database_i_cont_max": self.lineEdit_search_database_i_cont_max.text(),
                                         "lineEdit_search_database_switch_t_j_max_min": self.lineEdit_search_database_switch_t_j_max_min.text(),
                                         "lineEdit_search_database_switch_t_j_max_max": self.lineEdit_search_database_switch_t_j_max_max.text(),
                                         "lineEdit_search_database_diode_t_j_max_min": self.lineEdit_search_database_diode_t_j_max_min.text(),
                                         "lineEdit_search_database_diode_t_j_max_max": self.lineEdit_search_database_diode_t_j_max_max.text()}

        exporting_tools_settings_dict = {"comboBox_export_transistor": self.comboBox_export_transistor.currentText(),
                                         "lineEdit_export_number_parallel_transistors": self.lineEdit_export_number_parallel_transistors.text(),
                                         "lineEdit_export_simulink_v_supply": self.lineEdit_export_simulink_v_supply.text(),
                                         "lineEdit_export_simulink_r_g_on": self.lineEdit_export_simulink_r_g_on.text(),
                                         "lineEdit_export_simulink_r_g_off": self.lineEdit_export_simulink_r_g_off.text(),
                                         "lineEdit_export_simulink_normalize_t_to_v": self.lineEdit_export_simulink_normalize_t_to_v.text(),
                                         "lineEdit_export_gecko_v_supply": self.lineEdit_export_gecko_v_supply.text(),
                                         "lineEdit_export_gecko_r_g_on": self.lineEdit_export_gecko_r_g_on.text(),
                                         "lineEdit_export_gecko_r_g_off": self.lineEdit_export_gecko_r_g_off.text(),
                                         "lineEdit_export_gecko_v_g_on": self.lineEdit_export_gecko_v_g_on.text(),
                                         "lineEdit_export_gecko_v_g_off": self.lineEdit_export_gecko_v_g_off.text()}

        comparison_tools_settings_dict = {"comboBox_compare_transistor1": self.comboBox_compare_transistor1.currentText(),
                                          "comboBox_compare_transistor2": self.comboBox_compare_transistor2.currentText(),
                                          "comboBox_compare_transistor3": self.comboBox_compare_transistor3.currentText(),
                                          "comboBox_compare_v_g_on_transistor1": self.comboBox_compare_v_g_on_transistor1.currentText(),
                                          "comboBox_compare_v_g_off_transistor1": self.comboBox_compare_v_g_off_transistor1.currentText(),
                                          "comboBox_compare_v_g_on_transistor2": self.comboBox_compare_v_g_on_transistor2.currentText(),
                                          "comboBox_compare_v_g_off_transistor2": self.comboBox_compare_v_g_off_transistor2.currentText(),
                                          "comboBox_compare_v_g_on_transistor3": self.comboBox_compare_v_g_on_transistor3.currentText(),
                                          "comboBox_compare_v_g_off_transistor3": self.comboBox_compare_v_g_off_transistor3.currentText(),
                                          "comboBox_compare_plot1": self.comboBox_compare_plot1.currentText(),
                                          "comboBox_compare_plot2": self.comboBox_compare_plot2.currentText(),
                                          "comboBox_compare_plot3": self.comboBox_compare_plot3.currentText(),
                                          "comboBox_compare_plot4": self.comboBox_compare_plot4.currentText(),
                                          "comboBox_compare_plot5": self.comboBox_compare_plot5.currentText(),
                                          "comboBox_compare_plot6": self.comboBox_compare_plot6.currentText(),
                                          "comboBox_compare_plot7": self.comboBox_compare_plot7.currentText(),
                                          "comboBox_compare_plot8": self.comboBox_compare_plot8.currentText(),
                                          "comboBox_compare_plot9": self.comboBox_compare_plot9.currentText(),
                                          "lineEdit_compare_t_j_transistor1": self.lineEdit_compare_t_j_transistor1.text(),
                                          "lineEdit_compare_t_j_transistor2": self.lineEdit_compare_t_j_transistor2.text(),
                                          "lineEdit_compare_t_j_transistor3": self.lineEdit_compare_t_j_transistor3.text(),
                                          "lineEdit_compare_v_supply_transistor1": self.lineEdit_compare_v_supply_transistor1.text(),
                                          "lineEdit_compare_number_parallel_transistor1": self.lineEdit_compare_number_parallel_transistor1.text(),
                                          "lineEdit_compare_v_supply_transistor2": self.lineEdit_compare_v_supply_transistor2.text(),
                                          "lineEdit_compare_number_parallel_transistor2": self.lineEdit_compare_number_parallel_transistor2.text(),
                                          "lineEdit_compare_v_supply_transistor3": self.lineEdit_compare_v_supply_transistor3.text(),
                                          "lineEdit_compare_number_parallel_transistor3": self.lineEdit_compare_number_parallel_transistor3.text(),
                                          "label_compare_r_g_on_value_transistor1": self.label_compare_r_g_on_value_transistor1.text(),
                                          "label_compare_r_g_off_value_transistor1": self.label_compare_r_g_off_value_transistor1.text(),
                                          "label_compare_r_g_on_value_transistor2": self.label_compare_r_g_on_value_transistor2.text(),
                                          "label_compare_r_g_off_value_transistor2": self.label_compare_r_g_off_value_transistor2.text(),
                                          "label_compare_r_g_on_value_transistor3": self.label_compare_r_g_on_value_transistor3.text(),
                                          "label_compare_r_g_off_value_transistor3": self.label_compare_r_g_off_value_transistor3.text(),
                                          "slider_compare_r_g_on_transistor1": self.slider_compare_r_g_on_transistor1.value(),
                                          "slider_compare_r_g_off_transistor1": self.slider_compare_r_g_off_transistor1.value(),
                                          "slider_compare_r_g_on_transistor2": self.slider_compare_r_g_on_transistor2.value(),
                                          "slider_compare_r_g_off_transistor2": self.slider_compare_r_g_off_transistor2.value(),
                                          "slider_compare_r_g_on_transistor3": self.slider_compare_r_g_on_transistor3.value(),
                                          "slider_compare_r_g_off_transistor3": self.slider_compare_r_g_off_transistor3.value()}

        topology_calculator_settings_dict = {"comboBox_topology_topology": self.comboBox_topology_topology.currentText(),
                                             "comboBox_topology_transistor1": self.comboBox_topology_transistor1.currentText(),
                                             "comboBox_topology_transistor2": self.comboBox_topology_transistor2.currentText(),
                                             "comboBox_topology_v_g_on_transistor1": self.comboBox_topology_v_g_on_transistor1.currentText(),
                                             "comboBox_topology_plot1_line_contour": self.comboBox_topology_plot1_line_contour.currentText(),
                                             "comboBox_topology_plot1_x_axis": self.comboBox_topology_plot1_x_axis.currentText(),
                                             "comboBox_topology_plot1_y_axis": self.comboBox_topology_plot1_y_axis.currentText(),
                                             "comboBox_topology_plot1_z_axis": self.comboBox_topology_plot1_z_axis.currentText(),
                                             "comboBox_topology_plot2_line_contour": self.comboBox_topology_plot2_line_contour.currentText(),
                                             "comboBox_topology_plot2_x_axis": self.comboBox_topology_plot2_x_axis.currentText(),
                                             "comboBox_topology_plot2_y_axis": self.comboBox_topology_plot2_y_axis.currentText(),
                                             "comboBox_topology_plot2_z_axis": self.comboBox_topology_plot2_z_axis.currentText(),
                                             "comboBox_topology_plot3_line_contour": self.comboBox_topology_plot3_line_contour.currentText(),
                                             "comboBox_topology_plot3_x_axis": self.comboBox_topology_plot3_x_axis.currentText(),
                                             "comboBox_topology_plot3_y_axis": self.comboBox_topology_plot3_y_axis.currentText(),
                                             "comboBox_topology_plot3_z_axis": self.comboBox_topology_plot3_z_axis.currentText(),
                                             "comboBox_topology_plot4_line_contour": self.comboBox_topology_plot4_line_contour.currentText(),
                                             "comboBox_topology_plot4_x_axis": self.comboBox_topology_plot4_x_axis.currentText(),
                                             "comboBox_topology_plot4_y_axis": self.comboBox_topology_plot4_y_axis.currentText(),
                                             "comboBox_topology_plot4_z_axis": self.comboBox_topology_plot4_z_axis.currentText(),
                                             "comboBox_topology_plot5_line_contour": self.comboBox_topology_plot5_line_contour.currentText(),
                                             "comboBox_topology_plot5_x_axis": self.comboBox_topology_plot5_x_axis.currentText(),
                                             "comboBox_topology_plot5_y_axis": self.comboBox_topology_plot5_y_axis.currentText(),
                                             "comboBox_topology_plot5_z_axis": self.comboBox_topology_plot5_z_axis.currentText(),
                                             "comboBox_topology_plot6_line_contour": self.comboBox_topology_plot6_line_contour.currentText(),
                                             "comboBox_topology_plot6_x_axis": self.comboBox_topology_plot6_x_axis.currentText(),
                                             "comboBox_topology_plot6_y_axis": self.comboBox_topology_plot6_y_axis.currentText(),
                                             "comboBox_topology_plot6_z_axis": self.comboBox_topology_plot6_z_axis.currentText(),
                                             "lineEdit_topology_number_parallel_transistor1": self.lineEdit_topology_number_parallel_transistor1.text(),
                                             "lineEdit_topology_number_parallel_transistor2": self.lineEdit_topology_number_parallel_transistor2.text(),
                                             "lineEdit_topology_output_power": self.lineEdit_topology_output_power.text(),
                                             "lineEdit_topology_v_in": self.lineEdit_topology_v_in.text(),
                                             "lineEdit_topology_v_out": self.lineEdit_topology_v_out.text(),
                                             "lineEdit_topology_frequency": self.lineEdit_topology_frequency.text(),
                                             "lineEdit_topology_zeta": self.lineEdit_topology_zeta.text(),
                                             "lineEdit_topology_temperature_heatsink": self.lineEdit_topology_temperature_heatsink.text(),
                                             "lineEdit_topology_thermal_resistance_heatsink": self.lineEdit_topology_thermal_resistance_heatsink.text(),
                                             "lineEdit_topology_output_power_min": self.lineEdit_topology_output_power_min.text(),
                                             "lineEdit_topology_v_in_min": self.lineEdit_topology_v_in_min.text(),
                                             "lineEdit_topology_v_out_min": self.lineEdit_topology_v_out_min.text(),
                                             "lineEdit_topology_frequency_min": self.lineEdit_topology_frequency_min.text(),
                                             "lineEdit_topology_zeta_min": self.lineEdit_topology_zeta_min.text(),
                                             "lineEdit_topology_output_power_max": self.lineEdit_topology_output_power_max.text(),
                                             "lineEdit_topology_v_in_max": self.lineEdit_topology_v_in_max.text(),
                                             "lineEdit_topology_v_out_max": self.lineEdit_topology_v_out_max.text(),
                                             "lineEdit_topology_frequency_max": self.lineEdit_topology_frequency_max.text(),
                                             "lineEdit_topology_zeta_max": self.lineEdit_topology_zeta_max.text(),
                                             "label_topology_slider_r_g_on_value_transistor1": self.label_topology_slider_r_g_on_value_transistor1.text(),
                                             "label_topology_slider_r_g_off_value_transistor1": self.label_topology_slider_r_g_off_value_transistor1.text(),
                                             "slider_topology_r_g_on_transistor1": self.slider_topology_r_g_on_transistor1.value(),
                                             "slider_topology_r_g_off_transistor1": self.slider_topology_r_g_off_transistor1.value()}

        all_settings_dict = {**search_database_settings_dict, **exporting_tools_settings_dict, **comparison_tools_settings_dict, **topology_calculator_settings_dict}
        return all_settings_dict


    def export_settings(self):
        """
        Saves all user settings as json file to a directory selected by the user

        :return: None
        """
        all_settings_dict = self.get_settings_dict()
        try:
            path = QFileDialog.getExistingDirectory(self, caption="Open Directory")
            with open(path + "/settings.json", 'w') as fp:
                json.dump(all_settings_dict, fp)
            self.show_popup_message(
                f"Saved settings as settings.json to <a href={path}>{path}</a>!")
        except:
            pass

    def save_settings(self):
        """
        Saves all user settings to the current working directory

        :return: None
        """
        all_settings_dict = self.get_settings_dict()
        path = pathlib.Path.cwd()
        with open(path.joinpath(resource_path("settings.json")), 'w') as fp:
            json.dump(all_settings_dict, fp)
        self.show_popup_message("Settings saved succsessfully!")

    def load_settings(self):
        """
        Loads a json file selected by the user containing all settings and sets it using the function set_all_settings

        :return: None
        """
        try:
            path = QFileDialog.getOpenFileName(self, "Open File", "", resource_path("(*.json)"))
            with open(path[0], 'r') as fp:
                all_settings_dict = json.load(fp)
            self.set_all_settings(all_settings_dict)
            self.show_popup_message("Settings loaded successfully!")
        except:
            if path[0] != "":
                self.show_popup_message("Loading the settings failed! Check if the correct file was selected!")

    def load_local_settings(self):
        """
        Loads the settings.json file in the current working directory and sets all settings

        :return: None
        """
        try:
            with open(resource_path("settings.json"), 'r') as fp:
                all_settings_dict = json.load(fp)
            self.set_all_settings(all_settings_dict)
        except:
            pass


    def set_all_settings(self, all_settings_dict):
        """
        Sets all settings given in the dict all_settings_dict

        :param all_settings_dict: dict containing all settings
        :return: None
        """
        self.checkBox_search_database_id.setChecked(all_settings_dict["checkBox_search_database_id"])
        self.checkBox_search_database_name.setChecked(all_settings_dict["checkBox_search_database_name"])
        self.checkBox_search_database_type.setChecked(all_settings_dict["checkBox_search_database_type"])
        self.checkBox_search_database_author.setChecked(all_settings_dict["checkBox_search_database_author"])
        self.checkBox_search_database_technology.setChecked(all_settings_dict["checkBox_search_database_technology"])
        self.checkBox_search_database_template_version.setChecked(
            all_settings_dict["checkBox_search_database_template_version"])
        self.checkBox_search_database_template_date.setChecked(
            all_settings_dict["checkBox_search_database_template_date"])
        self.checkBox_search_database_creation_date.setChecked(
            all_settings_dict["checkBox_search_database_creation_date"])
        self.checkBox_search_database_last_modified.setChecked(
            all_settings_dict["checkBox_search_database_last_modified"])
        self.checkBox_search_database_comment.setChecked(all_settings_dict["checkBox_search_database_comment"])
        self.checkBox_search_database_datasheet_hyperlink.setChecked(
            all_settings_dict["checkBox_search_database_datasheet_hyperlink"])
        self.checkBox_search_database_datasheet_date.setChecked(
            all_settings_dict["checkBox_search_database_datasheet_date"])
        self.checkBox_search_database_datasheet_version.setChecked(
            all_settings_dict["checkBox_search_database_datasheet_version"])
        self.checkBox_search_database_manufacturer.setChecked(
            all_settings_dict["checkBox_search_database_manufacturer"])
        self.checkBox_search_database_housing_type.setChecked(
            all_settings_dict["checkBox_search_database_housing_type"])
        self.checkBox_search_database_housing_area.setChecked(
            all_settings_dict["checkBox_search_database_housing_area"])
        self.checkBox_search_database_cooling_area.setChecked(
            all_settings_dict["checkBox_search_database_cooling_area"])
        self.checkBox_search_database_t_c_max.setChecked(all_settings_dict["checkBox_search_database_t_c_max"])
        self.checkBox_search_database_r_g_int.setChecked(all_settings_dict["checkBox_search_database_r_g_int"])
        self.checkBox_search_database_r_g_on_recommended.setChecked(
            all_settings_dict["checkBox_search_database_r_g_on_recommended"])
        self.checkBox_search_database_r_g_off_recommended.setChecked(
            all_settings_dict["checkBox_search_database_r_g_off_recommended"])
        self.checkBox_search_database_c_oss_fix.setChecked(all_settings_dict["checkBox_search_database_c_oss_fix"])
        self.checkBox_search_database_c_iss_fix.setChecked(all_settings_dict["checkBox_search_database_c_iss_fix"])
        self.checkBox_search_database_c_rss_fix.setChecked(all_settings_dict["checkBox_search_database_c_rss_fix"])
        self.checkBox_search_database_r_th_cs.setChecked(all_settings_dict["checkBox_search_database_r_th_cs"])
        self.checkBox_search_database_r_th_switch_cs.setChecked(
            all_settings_dict["checkBox_search_database_r_th_switch_cs"])
        self.checkBox_search_database_r_th_diode_cs.setChecked(
            all_settings_dict["checkBox_search_database_r_th_diode_cs"])
        self.checkBox_search_database_v_abs_max.setChecked(all_settings_dict["checkBox_search_database_v_abs_max"])
        self.checkBox_search_database_i_abs_max.setChecked(all_settings_dict["checkBox_search_database_i_abs_max"])
        self.checkBox_search_database_i_cont.setChecked(all_settings_dict["checkBox_search_database_i_cont"])
        self.checkBox_search_database_switch_comment.setChecked(
            all_settings_dict["checkBox_search_database_switch_comment"])
        self.checkBox_search_database_switch_manufacturer.setChecked(
            all_settings_dict["checkBox_search_database_switch_manufacturer"])
        self.checkBox_search_database_switch_technology.setChecked(
            all_settings_dict["checkBox_search_database_switch_technology"])
        self.checkBox_search_database_switch_t_j_max.setChecked(
            all_settings_dict["checkBox_search_database_switch_t_j_max"])
        self.checkBox_search_database_diode_comment.setChecked(
            all_settings_dict["checkBox_search_database_diode_comment"])
        self.checkBox_search_database_diode_manufacturer.setChecked(
            all_settings_dict["checkBox_search_database_diode_manufacturer"])
        self.checkBox_search_database_diode_technology.setChecked(
            all_settings_dict["checkBox_search_database_diode_technology"])
        self.checkBox_search_database_diode_t_j_max.setChecked(
            all_settings_dict["checkBox_search_database_diode_t_j_max"])
        self.lineEdit_search_database_id.setText(all_settings_dict["lineEdit_search_database_id"])
        self.lineEdit_search_database_name.setText(all_settings_dict["lineEdit_search_database_name"])
        self.lineEdit_search_database_type.setText(all_settings_dict["lineEdit_search_database_type"])
        self.lineEdit_search_database_author.setText(all_settings_dict["lineEdit_search_database_author"])
        self.lineEdit_search_database_technology.setText(all_settings_dict["lineEdit_search_database_technology"])
        self.lineEdit_search_database_template_version.setText(
            all_settings_dict["lineEdit_search_database_template_version"])
        self.lineEdit_search_database_template_date.setText(
            all_settings_dict["lineEdit_search_database_template_date"])
        self.lineEdit_search_database_creation_date.setText(
            all_settings_dict["lineEdit_search_database_creation_date"])
        self.lineEdit_search_database_last_modified.setText(
            all_settings_dict["lineEdit_search_database_last_modified"])
        self.lineEdit_search_database_comment.setText(all_settings_dict["lineEdit_search_database_comment"])
        self.lineEdit_search_database_datasheet_hyperlink.setText(
            all_settings_dict["lineEdit_search_database_datasheet_hyperlink"])
        self.lineEdit_search_database_datasheet_date.setText(
            all_settings_dict["lineEdit_search_database_datasheet_date"])
        self.lineEdit_search_database_datasheet_version.setText(
            all_settings_dict["lineEdit_search_database_datasheet_version"])
        self.lineEdit_search_database_manufacturer.setText(all_settings_dict["lineEdit_search_database_manufacturer"])
        self.lineEdit_search_database_housing_type.setText(all_settings_dict["lineEdit_search_database_housing_type"])
        self.lineEdit_search_database_switch_comment.setText(
            all_settings_dict["lineEdit_search_database_switch_comment"])
        self.lineEdit_search_database_switch_manufacturer.setText(
            all_settings_dict["lineEdit_search_database_switch_manufacturer"])
        self.lineEdit_search_database_switch_technology.setText(
            all_settings_dict["lineEdit_search_database_switch_technology"])
        self.lineEdit_search_database_diode_comment.setText(
            all_settings_dict["lineEdit_search_database_diode_comment"])
        self.lineEdit_search_database_diode_manufacturer.setText(
            all_settings_dict["lineEdit_search_database_diode_manufacturer"])
        self.lineEdit_search_database_housing_area_min.setText(
            all_settings_dict["lineEdit_search_database_housing_area_min"])
        self.lineEdit_search_database_housing_area_max.setText(
            all_settings_dict["lineEdit_search_database_housing_area_max"])
        self.lineEdit_search_database_cooling_area_min.setText(
            all_settings_dict["lineEdit_search_database_cooling_area_min"])
        self.lineEdit_search_database_cooling_area_max.setText(
            all_settings_dict["lineEdit_search_database_cooling_area_max"])
        self.lineEdit_search_database_t_c_max_min.setText(all_settings_dict["lineEdit_search_database_t_c_max_min"])
        self.lineEdit_search_database_t_c_max_max.setText(all_settings_dict["lineEdit_search_database_t_c_max_max"])
        self.lineEdit_search_database_r_g_int_min.setText(all_settings_dict["lineEdit_search_database_r_g_int_min"])
        self.lineEdit_search_database_r_g_int_max.setText(all_settings_dict["lineEdit_search_database_r_g_int_max"])
        self.lineEdit_search_database_r_g_on_recommended_min.setText(
            all_settings_dict["lineEdit_search_database_r_g_on_recommended_min"])
        self.lineEdit_search_database_r_g_on_recommended_max.setText(
            all_settings_dict["lineEdit_search_database_r_g_on_recommended_max"])
        self.lineEdit_search_database_r_g_off_recommended_min.setText(
            all_settings_dict["lineEdit_search_database_r_g_off_recommended_min"])
        self.lineEdit_search_database_r_g_off_recommended_max.setText(
            all_settings_dict["lineEdit_search_database_r_g_off_recommended_max"])
        self.lineEdit_search_database_c_oss_fix_min.setText(
            all_settings_dict["lineEdit_search_database_c_oss_fix_min"])
        self.lineEdit_search_database_c_oss_fix_max.setText(
            all_settings_dict["lineEdit_search_database_c_oss_fix_max"])
        self.lineEdit_search_database_c_iss_fix_min.setText(
            all_settings_dict["lineEdit_search_database_c_iss_fix_min"])
        self.lineEdit_search_database_c_iss_fix_max.setText(
            all_settings_dict["lineEdit_search_database_c_iss_fix_max"])
        self.lineEdit_search_database_c_rss_fix_min.setText(
            all_settings_dict["lineEdit_search_database_c_rss_fix_min"])
        self.lineEdit_search_database_c_rss_fix_max.setText(
            all_settings_dict["lineEdit_search_database_c_rss_fix_max"])
        self.lineEdit_search_database_r_th_cs_min.setText(all_settings_dict["lineEdit_search_database_r_th_cs_min"])
        self.lineEdit_search_database_r_th_cs_max.setText(all_settings_dict["lineEdit_search_database_r_th_cs_max"])
        self.lineEdit_search_database_r_th_switch_cs_min.setText(
            all_settings_dict["lineEdit_search_database_r_th_switch_cs_min"])
        self.lineEdit_search_database_r_th_switch_cs_max.setText(
            all_settings_dict["lineEdit_search_database_r_th_switch_cs_max"])
        self.lineEdit_search_database_r_th_diode_cs_min.setText(
            all_settings_dict["lineEdit_search_database_r_th_diode_cs_min"])
        self.lineEdit_search_database_r_th_diode_cs_max.setText(
            all_settings_dict["lineEdit_search_database_r_th_diode_cs_max"])
        self.lineEdit_search_database_v_abs_max_min.setText(
            all_settings_dict["lineEdit_search_database_v_abs_max_min"])
        self.lineEdit_search_database_v_abs_max_max.setText(
            all_settings_dict["lineEdit_search_database_v_abs_max_max"])
        self.lineEdit_search_database_i_abs_max_min.setText(
            all_settings_dict["lineEdit_search_database_i_abs_max_min"])
        self.lineEdit_search_database_i_abs_max_max.setText(
            all_settings_dict["lineEdit_search_database_i_abs_max_max"])
        self.lineEdit_search_database_i_cont_min.setText(all_settings_dict["lineEdit_search_database_i_cont_min"])
        self.lineEdit_search_database_i_cont_max.setText(all_settings_dict["lineEdit_search_database_i_cont_max"])
        self.lineEdit_search_database_switch_t_j_max_min.setText(
            all_settings_dict["lineEdit_search_database_switch_t_j_max_min"])
        self.lineEdit_search_database_switch_t_j_max_max.setText(
            all_settings_dict["lineEdit_search_database_switch_t_j_max_max"])
        self.lineEdit_search_database_diode_t_j_max_min.setText(
            all_settings_dict["lineEdit_search_database_diode_t_j_max_min"])
        self.lineEdit_search_database_diode_t_j_max_max.setText(
            all_settings_dict["lineEdit_search_database_diode_t_j_max_max"])


        self.comboBox_export_transistor.setCurrentText(all_settings_dict["comboBox_export_transistor"])
        self.lineEdit_export_number_parallel_transistors.setText(
            all_settings_dict["lineEdit_export_number_parallel_transistors"])
        self.lineEdit_export_simulink_v_supply.setText(all_settings_dict["lineEdit_export_simulink_v_supply"])
        self.lineEdit_export_simulink_r_g_on.setText(all_settings_dict["lineEdit_export_simulink_r_g_on"])
        self.lineEdit_export_simulink_r_g_off.setText(all_settings_dict["lineEdit_export_simulink_r_g_off"])
        self.lineEdit_export_simulink_normalize_t_to_v.setText(
            all_settings_dict["lineEdit_export_simulink_normalize_t_to_v"])
        self.lineEdit_export_gecko_v_supply.setText(all_settings_dict["lineEdit_export_gecko_v_supply"])
        self.lineEdit_export_gecko_r_g_on.setText(all_settings_dict["lineEdit_export_gecko_r_g_on"])
        self.lineEdit_export_gecko_r_g_off.setText(all_settings_dict["lineEdit_export_gecko_r_g_off"])
        self.lineEdit_export_gecko_v_g_on.setText(all_settings_dict["lineEdit_export_gecko_v_g_on"])
        self.lineEdit_export_gecko_v_g_off.setText(all_settings_dict["lineEdit_export_gecko_v_g_off"])


        self.comboBox_compare_transistor1.setCurrentText(all_settings_dict["comboBox_compare_transistor1"])
        self.comboBox_compare_transistor2.setCurrentText(all_settings_dict["comboBox_compare_transistor2"])
        self.comboBox_compare_transistor3.setCurrentText(all_settings_dict["comboBox_compare_transistor3"])
        self.comboBox_compare_v_g_on_transistor1.setCurrentText(
            all_settings_dict["comboBox_compare_v_g_on_transistor1"])
        self.comboBox_compare_v_g_off_transistor1.setCurrentText(
            all_settings_dict["comboBox_compare_v_g_off_transistor1"])
        self.comboBox_compare_v_g_on_transistor2.setCurrentText(
            all_settings_dict["comboBox_compare_v_g_on_transistor2"])
        self.comboBox_compare_v_g_off_transistor2.setCurrentText(
            all_settings_dict["comboBox_compare_v_g_off_transistor2"])
        self.comboBox_compare_v_g_on_transistor3.setCurrentText(
            all_settings_dict["comboBox_compare_v_g_on_transistor3"])
        self.comboBox_compare_v_g_off_transistor3.setCurrentText(
            all_settings_dict["comboBox_compare_v_g_off_transistor3"])
        self.comboBox_compare_plot1.setCurrentText(all_settings_dict["comboBox_compare_plot1"])
        self.comboBox_compare_plot2.setCurrentText(all_settings_dict["comboBox_compare_plot2"])
        self.comboBox_compare_plot3.setCurrentText(all_settings_dict["comboBox_compare_plot3"])
        self.comboBox_compare_plot4.setCurrentText(all_settings_dict["comboBox_compare_plot4"])
        self.comboBox_compare_plot5.setCurrentText(all_settings_dict["comboBox_compare_plot5"])
        self.comboBox_compare_plot6.setCurrentText(all_settings_dict["comboBox_compare_plot6"])
        self.comboBox_compare_plot7.setCurrentText(all_settings_dict["comboBox_compare_plot7"])
        self.comboBox_compare_plot8.setCurrentText(all_settings_dict["comboBox_compare_plot8"])
        self.comboBox_compare_plot9.setCurrentText(all_settings_dict["comboBox_compare_plot9"])
        self.lineEdit_compare_t_j_transistor1.setText(all_settings_dict["lineEdit_compare_t_j_transistor1"])
        self.lineEdit_compare_t_j_transistor2.setText(all_settings_dict["lineEdit_compare_t_j_transistor2"])
        self.lineEdit_compare_t_j_transistor3.setText(all_settings_dict["lineEdit_compare_t_j_transistor3"])
        self.lineEdit_compare_v_supply_transistor1.setText(all_settings_dict["lineEdit_compare_v_supply_transistor1"])
        self.lineEdit_compare_number_parallel_transistor1.setText(
            all_settings_dict["lineEdit_compare_number_parallel_transistor1"])
        self.lineEdit_compare_v_supply_transistor2.setText(all_settings_dict["lineEdit_compare_v_supply_transistor2"])
        self.lineEdit_compare_number_parallel_transistor2.setText(
            all_settings_dict["lineEdit_compare_number_parallel_transistor2"])
        self.lineEdit_compare_v_supply_transistor3.setText(all_settings_dict["lineEdit_compare_v_supply_transistor3"])
        self.lineEdit_compare_number_parallel_transistor3.setText(
            all_settings_dict["lineEdit_compare_number_parallel_transistor3"])
        self.label_compare_r_g_on_value_transistor1.setText(
            all_settings_dict["label_compare_r_g_on_value_transistor1"])
        self.label_compare_r_g_off_value_transistor1.setText(
            all_settings_dict["label_compare_r_g_off_value_transistor1"])
        self.label_compare_r_g_on_value_transistor2.setText(
            all_settings_dict["label_compare_r_g_on_value_transistor2"])
        self.label_compare_r_g_off_value_transistor2.setText(
            all_settings_dict["label_compare_r_g_off_value_transistor2"])
        self.label_compare_r_g_on_value_transistor3.setText(
            all_settings_dict["label_compare_r_g_on_value_transistor3"])
        self.label_compare_r_g_off_value_transistor3.setText(
            all_settings_dict["label_compare_r_g_off_value_transistor3"])
        self.slider_compare_r_g_on_transistor1.setValue(int(all_settings_dict["slider_compare_r_g_on_transistor1"]))
        self.slider_compare_r_g_off_transistor1.setValue(int(all_settings_dict["slider_compare_r_g_off_transistor1"]))
        self.slider_compare_r_g_on_transistor2.setValue(int(all_settings_dict["slider_compare_r_g_on_transistor2"]))
        self.slider_compare_r_g_off_transistor2.setValue(int(all_settings_dict["slider_compare_r_g_off_transistor2"]))
        self.slider_compare_r_g_on_transistor3.setValue(int(all_settings_dict["slider_compare_r_g_on_transistor3"]))
        self.slider_compare_r_g_off_transistor3.setValue(int(all_settings_dict["slider_compare_r_g_off_transistor3"]))


        self.comboBox_topology_topology.setCurrentText(all_settings_dict["comboBox_topology_topology"])
        self.comboBox_topology_transistor1.setCurrentText(all_settings_dict["comboBox_topology_transistor1"])
        self.comboBox_topology_transistor2.setCurrentText(all_settings_dict["comboBox_topology_transistor2"])
        self.comboBox_topology_v_g_on_transistor1.setCurrentText(
            all_settings_dict["comboBox_topology_v_g_on_transistor1"])
        self.comboBox_topology_plot1_line_contour.setCurrentText(
            all_settings_dict["comboBox_topology_plot1_line_contour"])
        self.comboBox_topology_plot1_x_axis.setCurrentText(all_settings_dict["comboBox_topology_plot1_x_axis"])
        self.comboBox_topology_plot1_y_axis.setCurrentText(all_settings_dict["comboBox_topology_plot1_y_axis"])
        self.comboBox_topology_plot1_z_axis.setCurrentText(all_settings_dict["comboBox_topology_plot1_z_axis"])
        self.comboBox_topology_plot2_line_contour.setCurrentText(
            all_settings_dict["comboBox_topology_plot2_line_contour"])
        self.comboBox_topology_plot2_x_axis.setCurrentText(all_settings_dict["comboBox_topology_plot2_x_axis"])
        self.comboBox_topology_plot2_y_axis.setCurrentText(all_settings_dict["comboBox_topology_plot2_y_axis"])
        self.comboBox_topology_plot2_z_axis.setCurrentText(all_settings_dict["comboBox_topology_plot2_z_axis"])
        self.comboBox_topology_plot3_line_contour.setCurrentText(
            all_settings_dict["comboBox_topology_plot3_line_contour"])
        self.comboBox_topology_plot3_x_axis.setCurrentText(all_settings_dict["comboBox_topology_plot3_x_axis"])
        self.comboBox_topology_plot3_y_axis.setCurrentText(all_settings_dict["comboBox_topology_plot3_y_axis"])
        self.comboBox_topology_plot3_z_axis.setCurrentText(all_settings_dict["comboBox_topology_plot3_z_axis"])
        self.comboBox_topology_plot4_line_contour.setCurrentText(
            all_settings_dict["comboBox_topology_plot4_line_contour"])
        self.comboBox_topology_plot4_x_axis.setCurrentText(all_settings_dict["comboBox_topology_plot4_x_axis"])
        self.comboBox_topology_plot4_y_axis.setCurrentText(all_settings_dict["comboBox_topology_plot4_y_axis"])
        self.comboBox_topology_plot4_z_axis.setCurrentText(all_settings_dict["comboBox_topology_plot4_z_axis"])
        self.comboBox_topology_plot5_line_contour.setCurrentText(
            all_settings_dict["comboBox_topology_plot5_line_contour"])
        self.comboBox_topology_plot5_x_axis.setCurrentText(all_settings_dict["comboBox_topology_plot5_x_axis"])
        self.comboBox_topology_plot5_y_axis.setCurrentText(all_settings_dict["comboBox_topology_plot5_y_axis"])
        self.comboBox_topology_plot5_z_axis.setCurrentText(all_settings_dict["comboBox_topology_plot5_z_axis"])
        self.comboBox_topology_plot6_line_contour.setCurrentText(
            all_settings_dict["comboBox_topology_plot6_line_contour"])
        self.comboBox_topology_plot6_x_axis.setCurrentText(all_settings_dict["comboBox_topology_plot6_x_axis"])
        self.comboBox_topology_plot6_y_axis.setCurrentText(all_settings_dict["comboBox_topology_plot6_y_axis"])
        self.comboBox_topology_plot6_z_axis.setCurrentText(all_settings_dict["comboBox_topology_plot6_z_axis"])
        self.lineEdit_topology_number_parallel_transistor1.setText(
            all_settings_dict["lineEdit_topology_number_parallel_transistor1"])
        self.lineEdit_topology_number_parallel_transistor2.setText(
            all_settings_dict["lineEdit_topology_number_parallel_transistor2"])
        self.lineEdit_topology_output_power.setText(all_settings_dict["lineEdit_topology_output_power"])
        self.lineEdit_topology_v_in.setText(all_settings_dict["lineEdit_topology_v_in"])
        self.lineEdit_topology_v_out.setText(all_settings_dict["lineEdit_topology_v_out"])
        self.lineEdit_topology_frequency.setText(all_settings_dict["lineEdit_topology_frequency"])
        self.lineEdit_topology_zeta.setText(all_settings_dict["lineEdit_topology_zeta"])
        self.lineEdit_topology_temperature_heatsink.setText(
            all_settings_dict["lineEdit_topology_temperature_heatsink"])
        self.lineEdit_topology_thermal_resistance_heatsink.setText(
            all_settings_dict["lineEdit_topology_thermal_resistance_heatsink"])
        self.lineEdit_topology_output_power_min.setText(all_settings_dict["lineEdit_topology_output_power_min"])
        self.lineEdit_topology_v_in_min.setText(all_settings_dict["lineEdit_topology_v_in_min"])
        self.lineEdit_topology_v_out_min.setText(all_settings_dict["lineEdit_topology_v_out_min"])
        self.lineEdit_topology_frequency_min.setText(all_settings_dict["lineEdit_topology_frequency_min"])
        self.lineEdit_topology_zeta_min.setText(all_settings_dict["lineEdit_topology_zeta_min"])
        self.lineEdit_topology_output_power_max.setText(all_settings_dict["lineEdit_topology_output_power_max"])
        self.lineEdit_topology_v_in_max.setText(all_settings_dict["lineEdit_topology_v_in_max"])
        self.lineEdit_topology_v_out_max.setText(all_settings_dict["lineEdit_topology_v_out_max"])
        self.lineEdit_topology_frequency_max.setText(all_settings_dict["lineEdit_topology_frequency_max"])
        self.lineEdit_topology_zeta_max.setText(all_settings_dict["lineEdit_topology_zeta_max"])
        self.label_topology_slider_r_g_on_value_transistor1.setText(
            all_settings_dict["label_topology_slider_r_g_on_value_transistor1"])
        self.label_topology_slider_r_g_off_value_transistor1.setText(
            all_settings_dict["label_topology_slider_r_g_off_value_transistor1"])
        self.slider_topology_r_g_on_transistor1.setValue(int(all_settings_dict["slider_topology_r_g_on_transistor1"])),
        self.slider_topology_r_g_off_transistor1.setValue(int(all_settings_dict["slider_topology_r_g_off_transistor1"])),

    def clear_comparison_tools(self):
        """
        Clears numeric user inputs in Comparison Tools

        :return: None
        """
        self.comboBox_compare_v_g_on_transistor1.setCurrentIndex(0)
        self.comboBox_compare_v_g_off_transistor1.setCurrentIndex(0)
        self.comboBox_compare_v_g_on_transistor2.setCurrentIndex(0)
        self.comboBox_compare_v_g_off_transistor2.setCurrentIndex(0)
        self.comboBox_compare_v_g_on_transistor3.setCurrentIndex(0)
        self.comboBox_compare_v_g_off_transistor3.setCurrentIndex(0)
        self.lineEdit_compare_t_j_transistor1.clear()
        self.lineEdit_compare_t_j_transistor2.clear()
        self.lineEdit_compare_t_j_transistor3.clear()
        self.lineEdit_compare_v_supply_transistor1.clear()
        self.lineEdit_compare_number_parallel_transistor1.clear()
        self.lineEdit_compare_v_supply_transistor2.clear()
        self.lineEdit_compare_number_parallel_transistor2.clear()
        self.lineEdit_compare_v_supply_transistor3.clear()
        self.lineEdit_compare_number_parallel_transistor3.clear()
        self.label_compare_r_g_on_value_transistor1.setText(str(0.0))
        self.label_compare_r_g_off_value_transistor1.setText(str(0.0))
        self.label_compare_r_g_on_value_transistor2.setText(str(0.0))
        self.label_compare_r_g_off_value_transistor2.setText(str(0.0))
        self.label_compare_r_g_on_value_transistor3.setText(str(0.0))
        self.label_compare_r_g_off_value_transistor3.setText(str(0.0))
        self.slider_compare_r_g_on_transistor1.setValue(0)
        self.slider_compare_r_g_off_transistor1.setValue(0)
        self.slider_compare_r_g_on_transistor2.setValue(0)
        self.slider_compare_r_g_off_transistor2.setValue(0)
        self.slider_compare_r_g_on_transistor3.setValue(0)
        self.slider_compare_r_g_off_transistor3.setValue(0)

    def clear_topology_calculator(self):
        """
        Clears numeric user inputs in Topology Calculator

        :return:
        """
        self.lineEdit_topology_number_parallel_transistor1.clear()
        self.lineEdit_topology_number_parallel_transistor2.clear()
        self.lineEdit_topology_output_power.clear()
        self.lineEdit_topology_v_in.clear()
        self.lineEdit_topology_v_out.clear()
        self.lineEdit_topology_frequency.clear()
        self.lineEdit_topology_zeta.clear()
        self.lineEdit_topology_temperature_heatsink.clear()
        self.lineEdit_topology_thermal_resistance_heatsink.clear()
        self.lineEdit_topology_output_power_min.clear()
        self.lineEdit_topology_v_in_min.clear()
        self.lineEdit_topology_v_out_min.clear()
        self.lineEdit_topology_frequency_min.clear()
        self.lineEdit_topology_zeta_min.clear()
        self.lineEdit_topology_output_power_max.clear()
        self.lineEdit_topology_v_in_max.clear()
        self.lineEdit_topology_v_out_max.clear()
        self.lineEdit_topology_frequency_max.clear()
        self.lineEdit_topology_zeta_max.clear()
        self.label_topology_slider_r_g_on_value_transistor1.setText(str(0.0))
        self.label_topology_slider_r_g_off_value_transistor1.setText(str(0.0))
        self.slider_topology_r_g_on_transistor1.setValue(0)
        self.slider_topology_r_g_off_transistor1.setValue(0)

    ###CREATE TRANSISTOR###

    def fill_comboBoxes_create_transistor(self):
        """
        Fills the ComboBoxes containing the transistor types, housing types and manufacturers list

        :return: None
        """
        self.comboBox_create_transistor_add_data_dpt_dataset_type.addItems(["I_E Curve", "R_E Curve"])
        self.comboBox_create_transistor_add_data_dpt_energies.addItems(["both", "e_on", "e_off"])

        self.comboBox_create_transistor_switch_add_switching_losses_curve_type.addItems(["Switching Losses vs. Channel Current", "Switching Losses vs. Gate Resistor"])
        self.comboBox_create_transistor_diode_add_switching_losses_curve_type.addItems(["Switching Losses vs. Channel Current", "Switching Losses vs. Gate Resistor"])
        self.comboBox_create_transistor_switch_add_switching_losses_on_off.addItems(["E_on", "E_off"])

        self.comboBox_create_transistor_type.addItems(["MOSFET", "IGBT", "SiC-MOSFET", "GaN-Transistor"])


       # load housing types list and module manufacturers list from transistordatabase main path or from local path if not existent in main path
        if os.name == 'nt':
            # windows uses backslash
            path_transistordatabase = os.path.dirname(os.path.abspath(__file__)).replace("\gui", "")
        else:
            # linux uses forward slash
            path_transistordatabase = os.path.dirname(os.path.abspath(__file__)).replace("/gui", "")

        try:
            housing_types_list_file_path = os.path.join(path_transistordatabase, 'housing_types.txt')
            with open(housing_types_list_file_path, "r") as txt:
                housing_types_list = [line.replace("\n", "") for line in txt.readlines() if
                                      not line.startswith("#")]
        except:
            with open("housing_types.txt", "r") as txt:
                housing_types_list = [line.replace("\n", "") for line in txt.readlines() if
                                      not line.startswith("#")]

        self.comboBox_create_transistor_housing_type.addItems(housing_types_list)


        try:
            module_manufacturer_list_file_path = os.path.join(path_transistordatabase, 'module_manufacturers.txt')
            with open(module_manufacturer_list_file_path, "r") as txt:
                modulemanufacturer_list = [line.replace("\n", "") for line in txt.readlines() if
                                           not line.startswith("#")]
        except:
            with open("module_manufacturers.txt", "r") as txt:
                modulemanufacturer_list = [line.replace("\n", "") for line in txt.readlines() if
                                           not line.startswith("#")]

        self.comboBox_create_transistor_manufacturer.addItems(modulemanufacturer_list)
        self.comboBox_create_transistor_switch_manufacturer.addItems([""] + modulemanufacturer_list)
        self.comboBox_create_transistor_diode_manufacturer.addItems([""] + modulemanufacturer_list)
        self.comboBox_create_transistor_add_data_dpt_integration_interval.addItems([self.translation_dict['IEC 60747-9'], self.translation_dict['IEC 60747-8'], self.translation_dict['Mitsubishi'], self.translation_dict['Infineon'], self.translation_dict['Wolfspeed']])


    def clear_create_transistor(self):
        """
        Clears all inputs on the Create Transistor tab

        :return: None
        """
        for widget in self.scrollAreaWidgetContents_create_transistor.children():
            if isinstance(widget, QLineEdit) or isinstance(widget, QComboBox):
                widget.clear()
        self.fill_comboBoxes_create_transistor()



    def get_all_items_text_from_comboBox(self, comboBox):
        """
        Returns a list of all items stored in a ComboBox

        :param comboBox: comboBox object
        :return: list of all items text stored in a comboBox or empty list if comboBox is empty
        """
        if comboBox.count != 0:
            all_items_text = [comboBox.itemText(i) for i in range(comboBox.count())]
            return all_items_text
        else:
            return []


    def get_all_items_data_from_comboBox(self, comboBox):
        """
        Returns a list of all values connected to the texts in a comboBox

        :param comboBox: comboBox object
        :return: list of all values stored in a comboBox or empty list if comboBox is empty
        """
        if comboBox.count != 0:
            all_items_data = [comboBox.itemData(i) for i in range(comboBox.count())]
            return all_items_data
        else:
            return []


    def add_curve_switch_channel(self):
        """
        Adds an item to the comboBox to store the switch channel curves for a new transistor.
        Each comboBox item consists of a text containing the boundary conditions and path and
        a value that is a dictionary(graph key will be filled by the CurveChecker), which will be used to fill the transistor template

        :return: None
        """
        t_j = float(
            self.lineEdit_create_transistor_switch_add_channel_data_t_j.text()) if self.lineEdit_create_transistor_switch_add_channel_data_t_j.text().isnumeric() == True else None
        t_j_entry_name = self.lineEdit_create_transistor_switch_add_channel_data_t_j.text() + " °C" if self.lineEdit_create_transistor_switch_add_channel_data_t_j.text().isnumeric() == True else None

        v_g = float(
            self.lineEdit_create_transistor_switch_add_channel_data_v_g.text()) if self.lineEdit_create_transistor_switch_add_channel_data_v_g.text().isnumeric() == True else None
        v_g_entry_name = self.lineEdit_create_transistor_switch_add_channel_data_v_g.text() + " V" if self.lineEdit_create_transistor_switch_add_channel_data_v_g.text().isnumeric() == True else None

        file_path = self.browse_file_csv()

        comboBox_entry_name = f"T_j = {t_j_entry_name}, V_g = {v_g_entry_name}\nPath: {file_path}"
        data_dict = {"t_j": t_j, "v_g": v_g, "graph_v_i": "", "path": file_path}

        if file_path != "":
            all_items_text = self.get_all_items_text_from_comboBox(
                self.comboBox_create_transistor_switch_added_curves_channel_data)
            if comboBox_entry_name in all_items_text:
                self.show_popup_message("Curve already added!")
            else:
                self.comboBox_create_transistor_switch_added_curves_channel_data.addItem(comboBox_entry_name,
                                                                                         data_dict)

                self.comboBox_create_transistor_switch_added_curves_channel_data.setCurrentText(comboBox_entry_name)
                self.comboBox_create_transistor_switch_added_curves_channel_data.setDisabled(True)

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="Switch Channel Curve",
                    comboBox=self.comboBox_create_transistor_switch_added_curves_channel_data,
                    xlabel="Voltage in V",
                    ylabel="Current in A")


    def add_dpt_measurement_data_directory(self):
        """
        Adds double pulse test measurement data to a comboBox so that the data can be saved to transistor once "Load Transistor into Local Database" button is pressed

        :return: None
        """
        dataset_type = self.comboBox_create_transistor_add_data_dpt_dataset_type.currentText()
        comment = self.lineEdit_create_transistor_add_data_dpt_comment.text()
        load_inductance = float(
            self.lineEdit_create_transistor_add_data_dpt_load_inductance.text()) if self.lineEdit_create_transistor_add_data_dpt_load_inductance.text().isnumeric() == True else None
        commutation_inductance = float(
            self.lineEdit_create_transistor_add_data_commutation_inductance.text()) if self.lineEdit_create_transistor_add_data_commutation_inductance.text().isnumeric() == True else None
        measurement_date = self.lineEdit_create_transistor_add_data_dpt_measurement_date.text()
        measurement_testbench = self.comboBox_create_transistor_add_data_dpt_measurement_testbench.currentText()
        v_g = float(
            self.lineEdit_create_transistor_add_data_dpt_v_g.text()) if self.lineEdit_create_transistor_add_data_dpt_v_g.text().isnumeric() == True else None
        v_g_off = float(
            self.lineEdit_create_transistor_add_data_dpt_v_g_off.text()) if self.lineEdit_create_transistor_add_data_dpt_v_g_off.text().isnumeric() == True else None
        r_g_on = float(
            self.lineEdit_create_transistor_add_data_dpt_r_g_on.text()) if self.lineEdit_create_transistor_add_data_dpt_r_g_on.text().isnumeric() == True else None
        r_g_off = float(
            self.lineEdit_create_transistor_add_data_dpt_r_g_off.text()) if self.lineEdit_create_transistor_add_data_dpt_r_g_off.text().isnumeric() == True else None
        energies = self.comboBox_create_transistor_add_data_dpt_energies.currentText()
        commutation_device = self.lineEdit_create_transistor_add_data_dpt_commutation_device.text()
        integration_interval = self.comboBox_create_transistor_add_data_dpt_integration_interval.currentText()
        t_j = float(
            self.lineEdit_create_transistor_add_data_dpt_t_j.text()) if self.lineEdit_create_transistor_add_data_dpt_t_j.text().isnumeric() == True else None

        directory_path = QFileDialog.getExistingDirectory(self, caption="Open Directory")

        if directory_path != "":
            directory_path = directory_path + str("/*.csv")
            comboBox_entry_name = f"Directory: {directory_path}"

            if dataset_type == "I_E Curve":
                dataset_type = "graph_i_e"
            elif dataset_type == "R_E Curve":
                dataset_type = "graph_r_e"

            data_dict = {
                'path': directory_path,
                'dataset_type': dataset_type,
                'comment': comment,
                'load_inductance': load_inductance,
                'commutation_inductance': commutation_inductance,
                'measurement_date': measurement_date,
                'measurement_testbench': measurement_testbench,
                'v_g': v_g,
                'v_g_off': v_g_off,
                'r_g_on': r_g_on,
                'r_g_off': r_g_off,
                't_j': t_j,
                'energies': energies,
                'commutation_device': commutation_device,
                'integration_interval': list(self.translation_dict.keys())[list(self.translation_dict.values()).index(integration_interval)],
                'mode': 'save'}

            all_items_text = self.get_all_items_text_from_comboBox(
                self.comboBox_create_transistor_added_dpt)
            if comboBox_entry_name in all_items_text:
                self.show_popup_message("Data already added!")
            else:
                try:
                    new_dpt_dict = tdb.dpt_save_data(data_dict)
                    self.comboBox_create_transistor_added_dpt.addItem(comboBox_entry_name,
                                                                      {"new_dpt_dict": new_dpt_dict,
                                                                       "data_dict": data_dict})
                    self.comboBox_create_transistor_added_dpt.setCurrentText(comboBox_entry_name)
                except:
                    self.show_popup_message(f"Selected Directory is invalid! Possible Reasons <br>"
                                            f"1. Mandatory parameters: v_g_on, v_g_off, r_g_on, r_g_off, t_j <br>"
                                            f"2. Following keys in csv-file name must match <br>"
                                            f"    - Temperature <b>_</b>xx<b>C_</b><br>"
                                            f"    - Voltage <b>_</b>xxx<b>V_</b> <br>"
                                            f"    - Resistance <b>_</b>x.xx<b>R_</b> <br>"
                                            f"    - Current on/off <b> _ON_I</b> or <b> _OFF_I </b><br>"
                                            f"    - Voltage on/off <b>ON_U</b> or <b>OFF_U</b>")


    def view_dpt_measurement_data(self):
        """
        Shows added DPT measurement data in a ViewCurveWindow

        :return: None
        """
        try:
            data = self.comboBox_create_transistor_added_dpt.itemData(self.comboBox_create_transistor_added_dpt.currentIndex())
            data_dict = data["data_dict"]
            tdb.dpt_save_data(data_dict)
        except:
            try:
                data_dict = self.comboBox_create_transistor_added_dpt.itemData(
                    self.comboBox_create_transistor_added_dpt.currentIndex())
                if data_dict["dataset_type"] == "graph_i_e":
                    xlabel = "Current in A"
                elif data_dict["dataset_type"] == "graph_r_e":
                    xlabel = "Gate Resistor in Ω"

                self.ViewCurveWindow = ViewCurveWindow()
                self.ViewCurveWindow.view_curve(curve_title="DPT Measurement Data Curve",
                                                  xlabel=xlabel,
                                                  ylabel="Energy in J",
                                                  comboBox=self.comboBox_create_transistor_added_dpt)
            except:
                pass


    def delete_dpt_measurement_data(self):
        """
        Deletes added DPT measurement data

        :return: None
        """
        try:
            if "Directory" in self.comboBox_create_transistor_added_dpt.currentText():
                self.comboBox_create_transistor_added_dpt.removeItem(
                    self.comboBox_create_transistor_added_dpt.currentIndex())
            else:
                self.show_popup_message("Existing DPT Measurement Data can not be deleted!")
        except:
            pass


    def view_curve_switch_channel(self):
        """
        Runs CurveCecker for currently selected curve to edit it

        :return: None
        """
        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(curve_title="Switch Channel Curve",
                                           xlabel="Voltage in V",
                                           ylabel="Current in A",
                                           comboBox=self.comboBox_create_transistor_switch_added_curves_channel_data)

    def delete_curve_switch_channel(self):
        """
        Deletes the current item from the comboBox to store switch channel curves

        :return: None
        """
        self.comboBox_create_transistor_switch_added_curves_channel_data.removeItem(self.comboBox_create_transistor_switch_added_curves_channel_data.currentIndex())
        self.comboBox_create_transistor_switch_added_curves_channel_data.setDisabled(False)


    def add_curve_switch_switching_losses(self):
        t_j = float(
            self.lineEdit_create_transistor_switch_add_switching_losses_t_j.text()) if self.lineEdit_create_transistor_switch_add_switching_losses_t_j.text().isnumeric() == True else None
        t_j_entry_name = self.lineEdit_create_transistor_switch_add_switching_losses_t_j.text() + " °C" if self.lineEdit_create_transistor_switch_add_switching_losses_t_j.text().isnumeric() == True else None

        v_g = float(
            self.lineEdit_create_transistor_switch_add_switching_losses_v_g.text()) if self.lineEdit_create_transistor_switch_add_switching_losses_v_g.text().isnumeric() == True else None
        v_g_entry_name = self.lineEdit_create_transistor_switch_add_switching_losses_v_g.text() + " V" if self.lineEdit_create_transistor_switch_add_switching_losses_v_g.text().isnumeric() == True else None

        r_g = float(
            self.lineEdit_create_transistor_switch_add_switching_losses_r_g.text()) if self.lineEdit_create_transistor_switch_add_switching_losses_r_g.text().isnumeric() == True else None
        r_g_entry_name = self.lineEdit_create_transistor_switch_add_switching_losses_r_g.text() + " Ω" if self.lineEdit_create_transistor_switch_add_switching_losses_r_g.text().isnumeric() == True else None

        v_supply = float(
            self.lineEdit_create_transistor_switch_add_switching_losses_v_supply.text()) if self.lineEdit_create_transistor_switch_add_switching_losses_v_supply.text().isnumeric() == True else None
        v_supply_entry_name = self.lineEdit_create_transistor_switch_add_switching_losses_v_supply.text() + " V" if self.lineEdit_create_transistor_switch_add_switching_losses_v_supply.text().isnumeric() == True else None

        e_on_off = self.comboBox_create_transistor_switch_add_switching_losses_on_off.currentText()
        curve_type = self.comboBox_create_transistor_switch_add_switching_losses_curve_type.currentText()

        file_path = self.browse_file_csv()
        comboBox_entry_name = f"{e_on_off}: T_j = {t_j_entry_name}, V_g = {v_g_entry_name}, R_g = {r_g_entry_name}, V_supply = {v_supply_entry_name}\nPath: {file_path}"

        if curve_type == "Switching Losses vs. Channel Current":
            data_dict = {"e_on_off": e_on_off.lower(), "dataset_type": "graph_i_e", "t_j": t_j, 'v_g': v_g, 'v_supply': v_supply, 'r_g': r_g, "graph_i_e": "", "path": file_path}
        if curve_type == "Switching Losses vs. Gate Resistor":
            data_dict = {"e_on_off": e_on_off.lower(), "dataset_type": "graph_r_e", "t_j": t_j, 'v_g': v_g, 'v_supply': v_supply, 'r_g': r_g, "graph_r_e": "", "path": file_path}


        if file_path != "":
            all_items_text = self.get_all_items_text_from_comboBox(
                self.comboBox_create_transistor_switch_added_curves_switching_losses)
            if comboBox_entry_name in all_items_text:
                self.show_popup_message("Curve already added!")
            else:
                self.comboBox_create_transistor_switch_added_curves_switching_losses.addItem(comboBox_entry_name,
                                                                                             data_dict)

                self.comboBox_create_transistor_switch_added_curves_switching_losses.setCurrentText(
                    comboBox_entry_name)
                self.comboBox_create_transistor_switch_added_curves_switching_losses.setDisabled(True)

                if data_dict["dataset_type"] == "graph_i_e":
                    xlabel = "Current in A"
                elif data_dict["dataset_type"] == "graph_r_e":
                    xlabel = "Gate Resistor in Ω"

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="Switch Switching Losses Curve",
                    comboBox=self.comboBox_create_transistor_switch_added_curves_switching_losses,
                    xlabel=xlabel,
                    ylabel="Energy in J")

    def view_curve_switch_switching_losses(self):
        """
        Runs CurveCecker for currently selected curve to edit it

        :return: None
        """
        data_dict = self.comboBox_create_transistor_switch_added_curves_switching_losses.itemData(self.comboBox_create_transistor_switch_added_curves_switching_losses.currentIndex())
        if data_dict["dataset_type"] == "graph_i_e":
            xlabel= "Current in A"
        elif data_dict["dataset_type"] == "graph_r_e":
            xlabel= "Gate Resistor in Ω"

        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(curve_title="Switch Switching Losses Curve",
                                           xlabel=xlabel,
                                           ylabel="Energy in J",
                                           comboBox=self.comboBox_create_transistor_switch_added_curves_switching_losses)




    def delete_curve_switch_switching_losses(self):
        self.comboBox_create_transistor_switch_added_curves_switching_losses.removeItem(self.comboBox_create_transistor_switch_added_curves_switching_losses.currentIndex())
        self.comboBox_create_transistor_switch_added_curves_switching_losses.setDisabled(False)

    def add_curve_switch_gate_charge(self):
        i_channel = float(
            self.lineEdit_create_transistor_switch_add_gate_charge_i_channel.text()) if self.lineEdit_create_transistor_switch_add_gate_charge_i_channel.text() != "" else None
        i_channel_entry_name = self.lineEdit_create_transistor_switch_add_gate_charge_i_channel.text() + " A" if self.lineEdit_create_transistor_switch_add_gate_charge_i_channel.text() != "" else None

        i_g = float(
            self.lineEdit_create_transistor_switch_add_gate_charge_i_g.text()) if self.lineEdit_create_transistor_switch_add_gate_charge_i_g.text() != "" else None
        i_g_entry_name = self.lineEdit_create_transistor_switch_add_gate_charge_i_g.text() + " A" if self.lineEdit_create_transistor_switch_add_gate_charge_i_g.text() != "" else None

        t_j = float(
            self.lineEdit_create_transistor_switch_add_gate_charge_t_j.text()) if self.lineEdit_create_transistor_switch_add_gate_charge_t_j.text() != "" else None
        t_j_entry_name = self.lineEdit_create_transistor_switch_add_gate_charge_t_j.text() + " °C" if self.lineEdit_create_transistor_switch_add_gate_charge_t_j.text() != "" else None

        v_supply = float(
            self.lineEdit_create_transistor_switch_add_gate_charge_v_supply.text()) if self.lineEdit_create_transistor_switch_add_gate_charge_v_supply.text() != "" else None
        v_supply_entry_name = self.lineEdit_create_transistor_switch_add_gate_charge_v_supply.text() + " V" if self.lineEdit_create_transistor_switch_add_gate_charge_v_supply.text() != "" else None

        file_path = self.browse_file_csv()
        comboBox_entry_name = f"I_channel ={i_channel_entry_name}, T_j = {t_j_entry_name}, V_supply = {v_supply_entry_name}, I_g = {i_g_entry_name}\nPath: {file_path}"
        data_dict = {'i_channel': i_channel, 't_j': t_j, 'v_supply': v_supply, 'i_g': i_g, 'graph_q_v': "",
                     "path": file_path}

        if file_path != "":
            all_items_text = self.get_all_items_text_from_comboBox(
                self.comboBox_create_transistor_switch_added_curves_gate_charge)
            if comboBox_entry_name in all_items_text:
                self.show_popup_message("Curve already added!")
            else:
                self.comboBox_create_transistor_switch_added_curves_gate_charge.addItem(comboBox_entry_name,
                                                                                        data_dict)

                self.comboBox_create_transistor_switch_added_curves_gate_charge.setCurrentText(comboBox_entry_name)
                self.comboBox_create_transistor_switch_added_curves_gate_charge.setDisabled(True)

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="Switch Gate Charge Curve",
                    comboBox=self.comboBox_create_transistor_switch_added_curves_gate_charge,
                    xlabel="Gate Charge in nC",
                    ylabel="Gate Source Voltage in V")


    def view_curve_switch_gate_charge(self):
        """
        Runs CurveCecker for currently selected curve to edit it

        :return: None
        """
        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(
            curve_title="Switch Gate Charge Curve",
            comboBox=self.comboBox_create_transistor_switch_added_curves_gate_charge,
            xlabel="Gate Charge in nC",
            ylabel="Gate Source Voltage in V")


    def delete_curve_switch_gate_charge(self):
        self.comboBox_create_transistor_switch_added_curves_gate_charge.removeItem(
            self.comboBox_create_transistor_switch_added_curves_gate_charge.currentIndex())
        self.comboBox_create_transistor_switch_added_curves_gate_charge.setDisabled(False)


    def add_curve_switch_r_on(self):
        i_channel = float(
            self.lineEdit_create_transistor_switch_add_r_on_i_channel.text()) if self.lineEdit_create_transistor_switch_add_r_on_i_channel.text() != "" else None
        i_channel_entry_name = self.lineEdit_create_transistor_switch_add_r_on_i_channel.text() + " A" if self.lineEdit_create_transistor_switch_add_r_on_i_channel.text() != "" else None

        v_g = float(
            self.lineEdit_create_transistor_switch_add_r_on_v_g.text()) if self.lineEdit_create_transistor_switch_add_r_on_v_g.text() != "" else None
        v_g_entry_name = self.lineEdit_create_transistor_switch_add_r_on_v_g.text() + " V" if self.lineEdit_create_transistor_switch_add_r_on_v_g.text() != "" else None

        r_channel_nominal = float(
            self.lineEdit_create_transistor_switch_add_r_on_r_channel_nominal.text()) if self.lineEdit_create_transistor_switch_add_r_on_r_channel_nominal.text() != "" else None
        r_channel_nominal_entry_name = self.lineEdit_create_transistor_switch_add_r_on_r_channel_nominal.text() + " Ω" if self.lineEdit_create_transistor_switch_add_r_on_r_channel_nominal.text() != "" else None

        file_path = self.browse_file_csv()
        comboBox_entry_name = f"I_channel ={i_channel_entry_name}, V_g = {v_g_entry_name}, R_channel_nominal = {r_channel_nominal_entry_name}\nPath: {file_path}"
        data_dict = {'i_channel': i_channel, 'v_g': v_g, 'dataset_type': 't_r', 'r_channel_nominal': r_channel_nominal,
                     'graph_t_r': "", "path": file_path}

        if file_path != "":
            all_items_text = self.get_all_items_text_from_comboBox(
                self.comboBox_create_transistor_switch_added_curves_r_on)
            if comboBox_entry_name in all_items_text:
                self.show_popup_message("Curve already added!")
            else:
                self.comboBox_create_transistor_switch_added_curves_r_on.addItem(comboBox_entry_name,
                                                                                 data_dict)

                self.comboBox_create_transistor_switch_added_curves_r_on.setCurrentText(comboBox_entry_name)
                self.comboBox_create_transistor_switch_added_curves_r_on.setDisabled(True)

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="Switch On Resistance Curve",
                    comboBox=self.comboBox_create_transistor_switch_added_curves_r_on,
                    xlabel="Junction Temperature in °C",
                    ylabel="On Resistance in Ω")


    def view_curve_switch_r_on(self):
        """
        Runs CurveCecker for currently selected curve to edit it

        :return: None
        """
        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(
            curve_title="Switch On Resistance Curve",
            comboBox=self.comboBox_create_transistor_switch_added_curves_r_on,
            xlabel="Junction Temperature in °C",
            ylabel="On Resistance in Ω")

    def delete_curve_switch_r_on(self):
        self.comboBox_create_transistor_switch_added_curves_r_on.removeItem(
            self.comboBox_create_transistor_switch_added_curves_r_on.currentIndex())
        self.comboBox_create_transistor_switch_added_curves_r_on.setDisabled(False)


    def add_curve_diode_channel(self):
        t_j = float(
            self.lineEdit_create_transistor_diode_add_channel_data_t_j.text()) if self.lineEdit_create_transistor_diode_add_channel_data_t_j.text() != "" else None
        t_j_entry_name = self.lineEdit_create_transistor_diode_add_channel_data_t_j.text() + " °C" if self.lineEdit_create_transistor_diode_add_channel_data_t_j.text() != "" else None

        v_g = float(
            self.lineEdit_create_transistor_diode_add_channel_data_v_g.text()) if self.lineEdit_create_transistor_diode_add_channel_data_v_g.text() != "" else None
        v_g_entry_name = self.lineEdit_create_transistor_diode_add_channel_data_v_g.text() + " V" if self.lineEdit_create_transistor_diode_add_channel_data_v_g.text() != "" else None

        file_path = self.browse_file_csv()
        comboBox_entry_name = f"T_j = {t_j_entry_name}, V_g = {v_g_entry_name}\nPath: {file_path}"
        data_dict = {"t_j": t_j, 'v_g': v_g, "graph_v_i": "", "path": file_path}

        if file_path != "":
            all_items_text = self.get_all_items_text_from_comboBox(
                self.comboBox_create_transistor_diode_added_curves_channel_data)
            if comboBox_entry_name in all_items_text:
                self.show_popup_message("Curve already added!")
            else:
                self.comboBox_create_transistor_diode_added_curves_channel_data.addItem(comboBox_entry_name,
                                                                                        data_dict)

                self.comboBox_create_transistor_diode_added_curves_channel_data.setCurrentText(comboBox_entry_name)
                self.comboBox_create_transistor_diode_added_curves_channel_data.setDisabled(True)

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="Diode Channel Curve",
                    comboBox=self.comboBox_create_transistor_diode_added_curves_channel_data,
                    xlabel="Voltage in V",
                    ylabel="Current in A")


    def view_curve_diode_channel(self):
        """
        Runs CurveCecker for currently selected curve to edit it

        :return: None
        """
        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(
            curve_title="Diode Channel Curve",
            comboBox=self.comboBox_create_transistor_diode_added_curves_channel_data,
            xlabel="Voltage in V",
            ylabel="Current in A")



    def delete_curve_diode_channel(self):
        self.comboBox_create_transistor_diode_added_curves_channel_data.removeItem(self.comboBox_create_transistor_diode_added_curves_channel_data.currentIndex())
        self.comboBox_create_transistor_diode_added_curves_channel_data.setDisabled(False)


    def add_curve_diode_switching_losses(self):
        t_j = float(
            self.lineEdit_create_transistor_diode_add_switching_losses_t_j.text()) if self.lineEdit_create_transistor_diode_add_switching_losses_t_j.text() != "" else None
        t_j_entry_name = self.lineEdit_create_transistor_diode_add_switching_losses_t_j.text() + " °C" if self.lineEdit_create_transistor_diode_add_switching_losses_t_j.text() != "" else None

        v_g = float(
            self.lineEdit_create_transistor_diode_add_switching_losses_v_g.text()) if self.lineEdit_create_transistor_diode_add_switching_losses_v_g.text() != "" else None
        v_g_entry_name = self.lineEdit_create_transistor_diode_add_switching_losses_v_g.text() + " V" if self.lineEdit_create_transistor_diode_add_switching_losses_v_g.text() != "" else None

        r_g = float(
            self.lineEdit_create_transistor_diode_add_switching_losses_r_g.text()) if self.lineEdit_create_transistor_diode_add_switching_losses_r_g.text() != "" else None
        r_g_entry_name = self.lineEdit_create_transistor_diode_add_switching_losses_r_g.text() + " Ω" if self.lineEdit_create_transistor_diode_add_switching_losses_r_g.text() != "" else None

        v_supply = float(
            self.lineEdit_create_transistor_diode_add_switching_losses_v_supply.text()) if self.lineEdit_create_transistor_diode_add_switching_losses_v_supply.text() != "" else None
        v_supply_entry_name = self.lineEdit_create_transistor_diode_add_switching_losses_v_supply.text() + " V" if self.lineEdit_create_transistor_diode_add_switching_losses_v_supply.text() != "" else None

        file_path = self.browse_file_csv()
        comboBox_entry_name = f"T_j = {t_j_entry_name}, V_g = {v_g_entry_name}, R_g = {r_g_entry_name}, V_supply = {v_supply_entry_name}\nPath: {file_path}"

        if self.comboBox_create_transistor_diode_add_switching_losses_curve_type.currentText() == "Switching Losses vs. Channel Current":
            data_dict = {"dataset_type": "graph_i_e", "t_j": t_j, 'v_g': v_g, 'v_supply': v_supply, 'r_g': r_g, "graph_i_e": "", "path": file_path}
        elif self.comboBox_create_transistor_diode_add_switching_losses_curve_type.currentText() == "Switching Losses vs. Gate Resistor":
            data_dict = {"dataset_type": "graph_r_e", "t_j": t_j, 'v_g': v_g, 'v_supply': v_supply, 'r_g': r_g, "graph_r_e": "", "path": file_path}


        if file_path != "":
            all_items_text = self.get_all_items_text_from_comboBox(
                self.comboBox_create_transistor_diode_added_curves_switching_losses)
            if comboBox_entry_name in all_items_text:
                self.show_popup_message("Curve already added!")
            else:
                self.comboBox_create_transistor_diode_added_curves_switching_losses.addItem(comboBox_entry_name,
                                                                                            data_dict)

                self.comboBox_create_transistor_diode_added_curves_switching_losses.setCurrentText(
                    comboBox_entry_name)
                self.comboBox_create_transistor_diode_added_curves_switching_losses.setDisabled(True)

                if data_dict["dataset_type"] == "graph_i_e":
                    xlabel = "Current in A"
                elif data_dict["dataset_type"] == "graph_r_e":
                    xlabel = "Gate Resistor in Ω"

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="Diode Reverse Recovery Losses Curve",
                    comboBox=self.comboBox_create_transistor_diode_added_curves_switching_losses,
                    xlabel=xlabel,
                    ylabel="Energy in J")



    def view_curve_diode_switching_losses(self):
        """
        Runs CurveCecker for currently selected curve to edit it

        :return: None
        """
        data_dict = self.comboBox_create_transistor_diode_added_curves_switching_losses.itemData(self.comboBox_create_transistor_diode_added_curves_switching_losses.currentIndex())
        if data_dict["dataset_type"] == "graph_i_e":
            xlabel = "Current in A"
        elif data_dict["dataset_type"] == "graph_r_e":
            xlabel = "Gate Resistor in Ω"

        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(
            curve_title="Diode Reverse Recovery Losses Curve",
            comboBox=self.comboBox_create_transistor_diode_added_curves_switching_losses,
            xlabel=xlabel,
            ylabel="Energy in J")




    def delete_curve_diode_switching_losses(self):
        self.comboBox_create_transistor_diode_added_curves_switching_losses.removeItem(self.comboBox_create_transistor_diode_added_curves_switching_losses.currentIndex())
        self.comboBox_create_transistor_diode_added_curves_switching_losses.setDisabled(False)


    def add_curve_switch_soa_t_pulse(self):
        t_c = float(
            self.lineEdit_create_transistor_switch_soa_t_pulse_t_c.text()) if self.lineEdit_create_transistor_switch_soa_t_pulse_t_c.text() != "" else None
        t_c_entry_name = self.lineEdit_create_transistor_switch_soa_t_pulse_t_c.text() + " °C" if self.lineEdit_create_transistor_switch_soa_t_pulse_t_c.text() != "" else None

        time_pulse = float(
            self.lineEdit_create_transistor_switch_soa_t_pulse_time_pulse.text()) if self.lineEdit_create_transistor_switch_soa_t_pulse_time_pulse.text() != "" else None
        time_pulse_entry_name = self.lineEdit_create_transistor_switch_soa_t_pulse_time_pulse.text() + " s" if self.lineEdit_create_transistor_switch_soa_t_pulse_time_pulse.text() != "" else None

        file_path = self.browse_file_csv()
        comboBox_entry_name = f"T_j = {t_c_entry_name}, Time_pulse = {time_pulse_entry_name}\nPath: {file_path}"
        data_dict = {'t_c': t_c, 'time_pulse': time_pulse, 'graph_i_v': "", "path": file_path}

        if file_path != "":
            all_items_text = self.get_all_items_text_from_comboBox(
                self.comboBox_create_transistor_switch_added_curves_soa_t_pulse)
            if comboBox_entry_name in all_items_text:
                self.show_popup_message("Curve already added!")
            else:
                self.comboBox_create_transistor_switch_added_curves_soa_t_pulse.addItem(comboBox_entry_name,
                                                                                        data_dict)

                self.comboBox_create_transistor_switch_added_curves_soa_t_pulse.setCurrentText(comboBox_entry_name)
                self.comboBox_create_transistor_switch_added_curves_soa_t_pulse.setDisabled(True)

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="Switch SOA Curve",
                    comboBox=self.comboBox_create_transistor_switch_added_curves_soa_t_pulse,
                    xlabel="V_ds/V_r in V",
                    ylabel="I_d/I_r in A")

    def view_curve_switch_soa_t_pulse(self):
        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(
            curve_title="Switch SOA Curve",
            comboBox=self.comboBox_create_transistor_switch_added_curves_soa_t_pulse,
            xlabel="V_ds/V_r in V",
            ylabel="I_d/I_r in A")


    def delete_curve_switch_soa_t_pulse(self):
        self.comboBox_create_transistor_switch_added_curves_soa_t_pulse.removeItem(self.comboBox_create_transistor_switch_added_curves_soa_t_pulse.currentIndex())
        self.comboBox_create_transistor_switch_added_curves_soa_t_pulse.setDisabled(False)


    def add_curve_diode_soa_t_pulse(self):
        t_c = float(
            self.lineEdit_create_transistor_diode_soa_t_pulse_t_c.text()) if self.lineEdit_create_transistor_diode_soa_t_pulse_t_c.text() != "" else None
        t_c_entry_name = self.lineEdit_create_transistor_diode_soa_t_pulse_t_c.text() + " °C" if self.lineEdit_create_transistor_diode_soa_t_pulse_t_c.text() != "" else None

        time_pulse = float(
            self.lineEdit_create_transistor_diode_soa_t_pulse_time_pulse.text()) if self.lineEdit_create_transistor_diode_soa_t_pulse_time_pulse.text() != "" else None
        time_pulse_entry_name = self.lineEdit_create_transistor_diode_soa_t_pulse_time_pulse.text() + " s" if self.lineEdit_create_transistor_diode_soa_t_pulse_time_pulse.text() != "" else None

        file_path = self.browse_file_csv()
        comboBox_entry_name = f"T_j = {t_c_entry_name}, Time_pulse = {time_pulse_entry_name}\nPath: {file_path}"
        data_dict = {'t_c': t_c, 'time_pulse': time_pulse, 'graph_i_v': "", "path": file_path}

        if file_path != "":
            all_items_text = self.get_all_items_text_from_comboBox(
                self.comboBox_create_transistor_diode_added_curves_soa_t_pulse)
            if comboBox_entry_name in all_items_text:
                self.show_popup_message("Curve already added!")
            else:
                self.comboBox_create_transistor_diode_added_curves_soa_t_pulse.addItem(comboBox_entry_name,
                                                                                       data_dict)

                self.comboBox_create_transistor_diode_added_curves_soa_t_pulse.setCurrentText(comboBox_entry_name)
                self.comboBox_create_transistor_diode_added_curves_soa_t_pulse.setDisabled(True)

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="Diode SOA Curve",
                    comboBox=self.comboBox_create_transistor_diode_added_curves_soa_t_pulse,
                    xlabel="V_ds/V_r in V",
                    ylabel="I_d/I_r in A")



    def view_curve_diode_soa_t_pulse(self):
        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(
            curve_title="Diode SOA Curve",
            comboBox=self.comboBox_create_transistor_diode_added_curves_soa_t_pulse,
            xlabel="V_ds/V_r in V",
            ylabel="I_d/I_r in A")


    def delete_curve_diode_soa_t_pulse(self):
        self.comboBox_create_transistor_diode_added_curves_soa_t_pulse.removeItem(self.comboBox_create_transistor_diode_added_curves_soa_t_pulse.currentIndex())
        self.comboBox_create_transistor_diode_added_curves_soa_t_pulse.setDisabled(False)


    def add_curve_switch_t_rthjc(self):
        if self.comboBox_create_transistor_switch_added_curve_t_rthjc.count() == 0:
            file_path = self.browse_file_csv()
            comboBox_entry_name = f"Path: {file_path}"
            data_dict = {'graph_t_rthjc': "", "path": file_path}

            if file_path != "":
                self.comboBox_create_transistor_switch_added_curve_t_rthjc.addItem(comboBox_entry_name,
                                                                                   data_dict)

                self.comboBox_create_transistor_switch_added_curve_t_rthjc.setDisabled(True)

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="Switch T_Rthjc Curve",
                    comboBox=self.comboBox_create_transistor_switch_added_curve_t_rthjc,
                    xlabel="Junction Temperature in °C",
                    ylabel="T_Rthjc in Ω")
        else:
            self.show_popup_message("Curve has already been added!")


    def view_curve_switch_t_rthjc(self):
        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(
            curve_title="Switch T_Rthjc Curve",
            comboBox=self.comboBox_create_transistor_switch_added_curve_t_rthjc,
            xlabel="Junction Temperature in °C",
            ylabel="T_Rthjc in Ω")


    def delete_curve_switch_t_rthjc(self):
        self.comboBox_create_transistor_switch_added_curve_t_rthjc.clear()
        self.comboBox_create_transistor_diode_added_curves_soa_t_pulse.setDisabled(False)


    def add_curve_diode_t_rthjc(self):
        if self.comboBox_create_transistor_diode_added_curve_t_rthjc.count() == 0:
            file_path = self.browse_file_csv()
            comboBox_entry_name = f"Path: {file_path}"
            data_dict = {'graph_t_rthjc': "", "path": file_path}

            if file_path != "":
                self.comboBox_create_transistor_diode_added_curve_t_rthjc.addItem(comboBox_entry_name,
                                                                                  data_dict)

                self.comboBox_create_transistor_diode_added_curve_t_rthjc.setDisabled(True)

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="Diode T_Rthjc Curve",
                    comboBox=self.comboBox_create_transistor_diode_added_curve_t_rthjc,
                    xlabel="Junction Temperature in °C",
                    ylabel="T_Rthjc in Ω")
        else:
            self.show_popup_message("Curve has already been added!")

    def view_curve_diode_t_rthjc(self):
        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(
            curve_title="Diode T_Rthjc Curve",
            comboBox=self.comboBox_create_transistor_diode_added_curve_t_rthjc,
            xlabel="Junction Temperature in °C",
            ylabel="T_Rthjc in Ω")

    def delete_curve_diode_t_rthjc(self):
        self.comboBox_create_transistor_diode_added_curve_t_rthjc.clear()
        self.comboBox_create_transistor_diode_added_curve_t_rthjc.setDisabled(False)


    def add_curve_v_ecoss(self):
        if self.comboBox_create_transistor_added_curve_v_ecoss.count() == 0:
            file_path = self.browse_file_csv()
            comboBox_entry_name = f"Path: {file_path}"
            data_dict = {'graph_v_ecoss': "", "path": file_path}

            if file_path != "":
                self.comboBox_create_transistor_added_curve_v_ecoss.addItem(comboBox_entry_name,
                                                                            data_dict)

                self.comboBox_create_transistor_added_curve_v_ecoss.setDisabled(True)

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="E_coss Curve",
                    comboBox=self.comboBox_create_transistor_added_curve_v_ecoss,
                    xlabel="Voltage in V",
                    ylabel="Energy in J")
        else:
            self.show_popup_message("Curve has already been added!")

    def view_curve_v_ecoss(self):
        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(
            curve_title="E_coss Curve",
            comboBox=self.comboBox_create_transistor_added_curve_v_ecoss,
            xlabel="Voltage in V",
            ylabel="Energy in J")


    def delete_curve_v_ecoss(self):
        self.comboBox_create_transistor_added_curve_v_ecoss.clear()
        self.comboBox_create_transistor_added_curve_v_ecoss.setDisabled(False)



    def add_curve_c_iss_normal(self):
        if self.comboBox_create_transistor_added_c_iss_normal.count() == 0:
            t_j = float(
                self.lineEdit_create_transistor_add_curve_c_iss_t_j.text()) if self.lineEdit_create_transistor_add_curve_c_iss_t_j.text() != "" else None
            t_j_entry_name = self.lineEdit_create_transistor_add_curve_c_iss_t_j.text() + " °C" if self.lineEdit_create_transistor_add_curve_c_iss_t_j.text() != "" else None

            file_path = self.browse_file_csv()
            comboBox_entry_name = f"T_j = {t_j_entry_name}\nPath: {file_path}"
            data_dict = {"t_j": t_j, "graph_v_c": "", "path": file_path}

            if file_path != "":
                self.comboBox_create_transistor_added_c_iss_normal.addItem(comboBox_entry_name,
                                                                           data_dict)

                self.comboBox_create_transistor_added_c_iss_normal.setDisabled(True)

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="C_iss_normal Curve",
                    comboBox=self.comboBox_create_transistor_added_c_iss_normal,
                    xlabel="Voltage in V",
                    ylabel="Capacitance in F")
        else:
            self.show_popup_message("Curve has already been added!")

    def view_curve_c_iss_normal(self):
        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(
            curve_title="C_iss_normal Curve",
            comboBox=self.comboBox_create_transistor_added_c_iss_normal,
            xlabel="Voltage in V",
            ylabel="Capacitance in F")


    def delete_curve_c_iss_normal(self):
        self.comboBox_create_transistor_added_c_iss_normal.clear()
        self.comboBox_create_transistor_added_c_iss_normal.setDisabled(False)

    def add_curve_c_iss_detail(self):
        if self.comboBox_create_transistor_added_c_iss_detail.count() == 0:
            t_j = float(
                self.lineEdit_create_transistor_add_curve_c_iss_t_j.text()) if self.lineEdit_create_transistor_add_curve_c_iss_t_j.text() != "" else None
            t_j_entry_name = self.lineEdit_create_transistor_add_curve_c_iss_t_j.text() + " °C" if self.lineEdit_create_transistor_add_curve_c_iss_t_j.text() != "" else None

            file_path = self.browse_file_csv()
            comboBox_entry_name = f"T_j = {t_j_entry_name}\nPath: {file_path}"
            data_dict = {"t_j": t_j, "graph_v_c": "", "path": file_path}

            if file_path != "":
                self.comboBox_create_transistor_added_c_iss_detail.addItem(comboBox_entry_name,
                                                                           data_dict)

                self.comboBox_create_transistor_added_c_iss_detail.setDisabled(True)

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="C_iss_detail Curve",
                    comboBox=self.comboBox_create_transistor_added_c_iss_detail,
                    xlabel="Voltage in V",
                    ylabel="Capacitance in F")
        else:
            self.show_popup_message("Curve has already been added!")

    def view_curve_c_iss_detail(self):
        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(
            curve_title="C_iss_detail Curve",
            comboBox=self.comboBox_create_transistor_added_c_iss_detail,
            xlabel="Voltage in V",
            ylabel="Capacitance in F")

    def delete_curve_c_iss_detail(self):
        self.comboBox_create_transistor_added_c_iss_detail.clear()
        self.comboBox_create_transistor_added_c_iss_detail.setDisabled(False)

    def add_curve_c_oss_normal(self):
        if self.comboBox_create_transistor_added_c_oss_normal.count() == 0:
            t_j = float(
                self.lineEdit_create_transistor_add_curve_c_oss_t_j.text()) if self.lineEdit_create_transistor_add_curve_c_oss_t_j.text() != "" else None
            t_j_entry_name = self.lineEdit_create_transistor_add_curve_c_oss_t_j.text() + " °C" if self.lineEdit_create_transistor_add_curve_c_oss_t_j.text() != "" else None

            file_path = self.browse_file_csv()
            comboBox_entry_name = f"T_j = {t_j_entry_name}\nPath: {file_path}"
            data_dict = {"t_j": t_j, "graph_v_c": "", "path": file_path}

            if file_path != "":
                self.comboBox_create_transistor_added_c_oss_normal.addItem(comboBox_entry_name,
                                                                           data_dict)

                self.comboBox_create_transistor_added_c_oss_normal.setDisabled(True)

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="C_oss_normal Curve",
                    comboBox=self.comboBox_create_transistor_added_c_oss_normal,
                    xlabel="Voltage in V",
                    ylabel="Capacitance in F")
        else:
            self.show_popup_message("Curve has already been added!")

    def view_curve_c_oss_normal(self):
        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(
            curve_title="C_oss_normal Curve",
            comboBox=self.comboBox_create_transistor_added_c_oss_normal,
            xlabel="Voltage in V",
            ylabel="Capacitance in F")


    def delete_curve_c_oss_normal(self):
        self.comboBox_create_transistor_added_c_oss_normal.clear()
        self.comboBox_create_transistor_added_c_oss_normal.setDisabled(False)

    def add_curve_c_oss_detail(self):
        if self.comboBox_create_transistor_added_c_oss_detail.count() == 0:
            t_j = float(
                self.lineEdit_create_transistor_add_curve_c_oss_t_j.text()) if self.lineEdit_create_transistor_add_curve_c_oss_t_j.text() != "" else None
            t_j_entry_name = self.lineEdit_create_transistor_add_curve_c_oss_t_j.text() + " °C" if self.lineEdit_create_transistor_add_curve_c_oss_t_j.text() != "" else None

            file_path = self.browse_file_csv()
            comboBox_entry_name = f"T_j = {t_j_entry_name}\nPath: {file_path}"
            data_dict = {"t_j": t_j, "graph_v_c": "", "path": file_path}

            if file_path != "":
                self.comboBox_create_transistor_added_c_oss_detail.addItem(comboBox_entry_name,
                                                                           data_dict)

                self.comboBox_create_transistor_added_c_oss_detail.setDisabled(True)

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="C_oss_detail Curve",
                    comboBox=self.comboBox_create_transistor_added_c_oss_detail,
                    xlabel="Voltage in V",
                    ylabel="Capacitance in F")
        else:
            self.show_popup_message("Curve has already been added!")

    def view_curve_c_oss_detail(self):
        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(
            curve_title="C_oss_detail Curve",
            comboBox=self.comboBox_create_transistor_added_c_oss_detail,
            xlabel="Voltage in V",
            ylabel="Capacitance in F")

    def delete_curve_c_oss_detail(self):
        self.comboBox_create_transistor_added_c_oss_detail.clear()
        self.comboBox_create_transistor_added_c_oss_detail.setDisabled(False)

    def add_curve_c_rss_normal(self):
        if self.comboBox_create_transistor_added_c_rss_normal.count() == 0:
            t_j = float(
                self.lineEdit_create_transistor_add_curve_c_rss_t_j.text()) if self.lineEdit_create_transistor_add_curve_c_rss_t_j.text() != "" else None
            t_j_entry_name = self.lineEdit_create_transistor_add_curve_c_rss_t_j.text() + " °C" if self.lineEdit_create_transistor_add_curve_c_rss_t_j.text() != "" else None

            file_path = self.browse_file_csv()
            comboBox_entry_name = f"T_j = {t_j_entry_name}, Path\n{file_path}"
            data_dict = {"t_j": t_j, "graph_v_c": "", "path": file_path}

            if file_path != "":
                self.comboBox_create_transistor_added_c_rss_normal.addItem(comboBox_entry_name,
                                                                           data_dict)

                self.comboBox_create_transistor_added_c_rss_normal.setDisabled(True)

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="C_rss_normal Curve",
                    comboBox=self.comboBox_create_transistor_added_c_rss_normal,
                    xlabel="Voltage in V",
                    ylabel="Capacitance in F")
        else:
            self.show_popup_message("Curve has already been added!")

    def view_curve_c_rss_normal(self):
        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(
            curve_title="C_rss_normal Curve",
            comboBox=self.comboBox_create_transistor_added_c_rss_normal,
            xlabel="Voltage in V",
            ylabel="Capacitance in F")


    def delete_curve_c_rss_normal(self):
        self.comboBox_create_transistor_added_c_rss_normal.clear()
        self.comboBox_create_transistor_added_c_rss_normal.setDisabled(False)

    def add_curve_c_rss_detail(self):
        if self.comboBox_create_transistor_added_c_rss_detail.count() == 0:
            t_j = float(
                self.lineEdit_create_transistor_add_curve_c_rss_t_j.text()) if self.lineEdit_create_transistor_add_curve_c_rss_t_j.text() != "" else None
            t_j_entry_name = self.lineEdit_create_transistor_add_curve_c_rss_t_j.text() + " °C" if self.lineEdit_create_transistor_add_curve_c_rss_t_j.text() != "" else None

            file_path = self.browse_file_csv()
            comboBox_entry_name = f"T_j = {t_j_entry_name}\nPath: {file_path}"
            data_dict = {"t_j": t_j, "graph_v_c": "", "path": file_path}

            if file_path != "":
                self.comboBox_create_transistor_added_c_rss_detail.addItem(comboBox_entry_name,
                                                                           data_dict)

                self.comboBox_create_transistor_added_c_rss_detail.setDisabled(True)

                self.CurveCheckerWindow = CurveCheckerWindow()
                self.CurveCheckerWindow.run_curve_checker(
                    curve_title="C_rss_detail Curve",
                    comboBox=self.comboBox_create_transistor_added_c_rss_detail,
                    xlabel="Voltage in V",
                    ylabel="Capacitance in F")
        else:
            self.show_popup_message("Curve has already been added!")

    def view_curve_c_rss_detail(self):
        self.ViewCurveWindow = ViewCurveWindow()
        self.ViewCurveWindow.view_curve(
            curve_title="C_rss_detail Curve",
            comboBox=self.comboBox_create_transistor_added_c_rss_detail,
            xlabel="Voltage in V",
            ylabel="Capacitance in F")

    def delete_curve_c_rss_detail(self):
        self.comboBox_create_transistor_added_c_rss_detail.clear()
        self.comboBox_create_transistor_added_c_rss_detail.setDisabled(False)


    def browse_file_csv(self):
        """
        Opens up a window to browse a csv file
        :return: List containing the file path and type of the selected file
        """
        path = QFileDialog.getOpenFileName(self, "Open File", "", "(*.csv)")
        return path[0]


    def create_transistor(self):
        """
        Fills the transistor template with all inputs from the Create Transistor tab and creates a transistor object.
        Missing inputs are set to None or empty lists in case of curves.

        :return: transistor object
        """

        ###TRANSISTOR PARAMETERS###

        try:
            c_iss_normal_dict = self.comboBox_create_transistor_added_c_iss_normal.itemData(self.comboBox_create_transistor_added_c_iss_normal.currentIndex())
            c_iss_normal = c_iss_normal_dict["graph_v_c"]

            c_iss_detail_dict = self.comboBox_create_transistor_added_c_iss_detail.itemData(self.comboBox_create_transistor_added_c_iss_detail.currentIndex())
            c_iss_detail = c_iss_detail_dict["graph_v_c"]

            c_iss_merged = tdb.merge_curve(c_iss_normal, c_iss_detail)

            c_iss = {"t_j": float(c_iss_normal_dict["t_j"]),
                     "graph_v_c": c_iss_merged}
        except:
            try:
                c_iss = c_iss_normal_dict
            except:
                c_iss = None

        try:
            c_oss_normal_dict = self.comboBox_create_transistor_added_c_oss_normal.itemData(self.comboBox_create_transistor_added_c_oss_normal.currentIndex())
            c_oss_normal = c_oss_normal_dict["graph_v_c"]

            c_oss_detail_dict = self.comboBox_create_transistor_added_c_oss_detail.itemData(self.comboBox_create_transistor_added_c_oss_detail.currentIndex())
            c_oss_detail = c_oss_detail_dict["graph_v_c"]

            c_oss_merged = tdb.merge_curve(c_oss_normal, c_oss_detail)

            c_oss = {"t_j": float(c_oss_normal_dict["t_j"]),
                     "graph_v_c": c_oss_merged}
        except:
            try:
                c_oss = c_oss_normal_dict
            except:
                c_oss = None


        try:
            c_rss_normal_dict = self.comboBox_create_transistor_added_c_rss_normal.itemData(self.comboBox_create_transistor_added_c_rss_normal.currentIndex())
            c_rss_normal = c_iss_normal_dict["graph_v_c"]

            c_rss_detail_dict = self.comboBox_create_transistor_added_c_rss_detail.itemData(self.comboBox_create_transistor_added_c_rss_detail.currentIndex())
            c_rss_detail = c_rss_detail_dict["graph_v_c"]

            c_rss_merged = tdb.merge_curve(c_rss_normal, c_rss_detail)

            c_rss = {"t_j": float(c_rss_normal_dict["t_j"]),
                     "graph_v_c": c_rss_merged}
        except:
            try:
                c_rss = c_rss_normal_dict
            except:
                c_rss = None



        try:
            c_oss_er = {"c_o": float(self.lineEdit_create_transistor_transistor_c_oss_er_c_o.text()),
                        "v_gs": float(self.lineEdit_create_transistor_transistor_c_oss_er_v_gs.text()),
                        "v_ds": float(self.lineEdit_create_transistor_transistor_c_oss_er_v_ds.text())}
            c_oss_tr = {"c_o": float(self.lineEdit_create_transistor_transistor_c_oss_tr_c_o.text()),
                        "v_gs": float(self.lineEdit_create_transistor_transistor_c_oss_tr_v_gs.text()),
                        "v_ds": float(self.lineEdit_create_transistor_transistor_c_oss_tr_v_ds.text())}
        except:
            c_oss_er = None
            c_oss_tr = None

        try:
            v_ecoss_dict = self.comboBox_create_transistor_added_curve_v_ecoss.itemData(self.comboBox_create_transistor_added_curve_v_ecoss.currentIndex())
            v_ecoss = v_ecoss_dict["graph_v_ecoss"]
        except:
            v_ecoss = None

        try:
            housing_area = float(self.lineEdit_create_transistor_housing_area.text())
        except: housing_area = None

        try:
            cooling_area = float(self.lineEdit_create_transistor_cooling_area.text())
        except: cooling_area = None

        try:
            v_abs_max = float(self.lineEdit_create_transistor_v_abs_max.text())
        except: v_abs_max = None

        try:
            i_abs_max = float(self.lineEdit_create_transistor_i_abs_max.text())
        except: i_abs_max = None

        try:
            i_cont = float(self.lineEdit_create_transistor_i_cont.text())
        except: i_cont = None

        try:
            c_iss_fix = float(self.lineEdit_create_transistor_c_iss_fix.text())
        except: c_iss_fix = None

        try:
            c_oss_fix = float(self.lineEdit_create_transistor_c_oss_fix.text())
        except: c_oss_fix = None

        try:
            c_rss_fix = float(self.lineEdit_create_transistor_c_rss_fix.text())
        except: c_rss_fix = None

        try:
            r_g_int = float(self.lineEdit_create_transistor_r_g_int.text())
        except: r_g_int = None

        try:
            r_th_cs = float(self.lineEdit_create_transistor_r_th_cs.text())
        except: r_th_cs = None

        try:
            r_th_diode_cs = float(self.lineEdit_create_transistor_r_th_diode_cs.text())
        except: r_th_diode_cs = None

        try:
            r_th_switch_cs = float(self.lineEdit_create_transistor_r_th_switch_cs.text())
        except: r_th_switch_cs = None

        try:
            r_g_on_recommended = float(self.lineEdit_create_transistor_r_g_on_recommended.text())
        except: r_g_on_recommended = None

        try:
            r_g_off_recommended = float(self.lineEdit_create_transistor_r_g_off_recommended.text())
        except: r_g_off_recommended = None

        try:
            t_c_max = float(self.lineEdit_create_transistor_t_c_max.text())
        except: t_c_max = None


        #get currently existing dpt data
        raw_measurement_data = None
        e_on_meas = None
        e_off_meas = None
        for i in range(self.comboBox_create_transistor_added_dpt.count()):
            if self.comboBox_create_transistor_added_dpt.itemText(i) == "All DPT Measurement Data":
                dpt_measurement_data = self.comboBox_create_transistor_added_dpt.itemData(i)
                raw_measurement_data = dpt_measurement_data["raw_measurement_data"]
                e_on_meas = dpt_measurement_data["e_on_meas"]
                e_off_meas = dpt_measurement_data["e_off_meas"]



        transistor_args = {'name': self.lineEdit_create_transistor_name.text(),
                           'type': self.comboBox_create_transistor_type.currentText(),
                           'author': self.lineEdit_create_transistor_author.text(),
                           'comment': self.lineEdit_create_transistor_comment.text(),
                           'manufacturer': self.comboBox_create_transistor_manufacturer.currentText(),
                           'datasheet_hyperlink': self.lineEdit_create_transistor_datasheet_hyperlink.text(),
                           'datasheet_date': self.lineEdit_create_transistor_datasheet_date.text(),
                           'datasheet_version': self.lineEdit_create_transistor_datasheet_version.text(),
                           'housing_area': housing_area,
                           'cooling_area': cooling_area,
                           'housing_type': self.comboBox_create_transistor_housing_type.currentText(),
                           'v_abs_max': v_abs_max,
                           'i_abs_max': i_abs_max,
                           'i_cont': i_cont,
                           'c_iss': c_iss,
                           'c_oss': c_oss,
                           'c_rss': c_rss,
                           'c_oss_er': c_oss_er,
                           'c_oss_tr': c_oss_tr,
                           'c_iss_fix': c_iss_fix,
                           'c_oss_fix': c_oss_fix,
                           'c_rss_fix': c_rss_fix,
                           'graph_v_ecoss': v_ecoss,
                           'r_g_int': r_g_int,
                           'r_th_cs': r_th_cs,
                           'r_th_diode_cs': r_th_diode_cs,
                           'r_th_switch_cs': r_th_switch_cs,
                           "r_g_on_recommended": r_g_on_recommended,
                           "r_g_off_recommended": r_g_off_recommended,
                           "t_c_max": t_c_max,
                           "raw_measurement_data": raw_measurement_data}


        ###SWITCH PARAMETERS#
        try:
            t_rthjc_dict = self.comboBox_create_transistor_switch_added_curve_t_rthjc.itemData(self.comboBox_create_transistor_switch_added_curve_t_rthjc.currentIndex())
            t_rthjc = t_rthjc_dict["graph_t_rthjc"]
        except: t_rthjc = None

        try:
            r_th_total_switch = float(self.lineEdit_create_transistor_switch_r_th_total.text())
        except: r_th_total_switch = None

        try:
            c_th_total = float(self.lineEdit_create_transistor_switch_c_th_total.text())
        except: c_th_total = None

        try:
            tau_total = float(self.lineEdit_create_transistor_switch_tau_total.text())
        except: tau_total = None

        switch_foster_args = {
            'r_th_vector': list(map(float, self.lineEdit_create_transistor_switch_r_th_vector.text().split())),
            'r_th_total': r_th_total_switch,
            'c_th_vector': list(map(float, self.lineEdit_create_transistor_switch_c_th_vector.text().split())),
            'c_th_total': c_th_total,
            'tau_vector': list(map(float, self.lineEdit_create_transistor_switch_tau_vector.text().split())),
            'tau_total': tau_total,
            'graph_t_rthjc': t_rthjc}

        for key in switch_foster_args.keys():
            if switch_foster_args[key] == [] or switch_foster_args[key] == "":
                switch_foster_args[key] = None


        e_on_off_list = self.get_all_items_data_from_comboBox(self.comboBox_create_transistor_switch_added_curves_switching_losses)
        e_on_list = []
        e_off_list = []
        for i in range(len(e_on_off_list)):
            if e_on_off_list[i]["e_on_off"] == "e_on":
                del e_on_off_list[i]["e_on_off"]
                e_on_list.append(e_on_off_list[i])
            elif e_on_off_list[i]["e_on_off"] == "e_off":
                del e_on_off_list[i]["e_on_off"]
                e_off_list.append(e_on_off_list[i])

        try:
            t_j_max_switch = float(self.lineEdit_create_transistor_switch_t_j_max.text())
        except: t_j_max_switch = None

        switch_args = {
            'comment': self.lineEdit_create_transistor_switch_comment.text(),
            'manufacturer': self.comboBox_create_transistor_switch_manufacturer.currentText(),
            'technology': self.lineEdit_create_transistor_switch_technology.text(),
            't_j_max': t_j_max_switch,
            'channel': self.get_all_items_data_from_comboBox(
                self.comboBox_create_transistor_switch_added_curves_channel_data),
            'e_on': e_on_list,
            'e_off': e_off_list,
            'charge_curve': self.get_all_items_data_from_comboBox(
                self.comboBox_create_transistor_switch_added_curves_gate_charge),
            'r_channel_th': self.get_all_items_data_from_comboBox(
                self.comboBox_create_transistor_switch_added_curves_r_on),
            'thermal_foster': switch_foster_args,
            'soa': self.get_all_items_data_from_comboBox(self.comboBox_create_transistor_switch_added_curves_soa_t_pulse),
            "e_on_meas": e_on_meas,
            "e_off_meas": e_off_meas}


        ###DIODE PARAMATERS###
        try:
            t_rthjc_dict = self.comboBox_create_transistor_diode_added_curve_t_rthjc.itemData(self.comboBox_create_transistor_diode_added_curve_t_rthjc.currentIndex())
            t_rthjc = t_rthjc_dict["graph_t_rthjc"]
        except: t_rthjc = None

        try:
            r_th_total_diode = float(self.lineEdit_create_transistor_diode_r_th_total.text())
        except: r_th_total_diode = None

        try:
            c_th_total = float(self.lineEdit_create_transistor_diode_c_th_total.text())
        except: c_th_total = None

        try:
            tau_total=float(self.lineEdit_create_transistor_diode_tau_total.text())
        except: tau_total = None


        diode_foster_args = {
            'r_th_vector': list(map(float, self.lineEdit_create_transistor_diode_r_th_vector.text().split())),
            'r_th_total': r_th_total_diode,
            'c_th_vector': list(map(float, self.lineEdit_create_transistor_diode_c_th_vector.text().split())),
            'c_th_total': c_th_total,
            'tau_vector': list(map(float, self.lineEdit_create_transistor_diode_tau_vector.text().split())),
            'tau_total': tau_total,
            'graph_t_rthjc': t_rthjc}

        for key in list(diode_foster_args.keys()):
            if diode_foster_args[key] == [] or diode_foster_args[key] == "":
                diode_foster_args[key] = None

        try:
            t_j_max_diode = float(self.lineEdit_create_transistor_diode_t_j_max.text())
        except: t_j_max_diode = None

        diode_args = {'comment': self.lineEdit_create_transistor_diode_comment.text(),
                      'manufacturer': self.comboBox_create_transistor_diode_manufacturer.currentText(),
                      'technology': self.lineEdit_create_transistor_diode_technology.text(),
                      't_j_max': t_j_max_diode,
                      'channel': self.get_all_items_data_from_comboBox(
                          self.comboBox_create_transistor_diode_added_curves_channel_data),
                      'e_rr': self.get_all_items_data_from_comboBox(
                          self.comboBox_create_transistor_diode_added_curves_switching_losses),
                      'thermal_foster': diode_foster_args,
                      'soa': self.get_all_items_data_from_comboBox(self.comboBox_create_transistor_diode_added_curves_soa_t_pulse)}

        transistor = tdb.Transistor(transistor_args, switch_args, diode_args)

        #Add new DPT data to new transistor object
        data_list = self.get_all_items_data_from_comboBox(self.comboBox_create_transistor_added_dpt)
        for data in data_list:
            try:
                transistor.add_dpt_measurement(data["new_dpt_dict"])
            except:
                pass

        dict = {"transistor_args": transistor_args, "switch_args": switch_args, "diode_args": diode_args, "transistor": transistor}


        return dict



    def load_transistor_into_local_database(self):
        """
        Checks if transistor already exists and opens an information window to ask user if he wants to overwrite the transistor,
        else saves the new transistor to local database

        :return: None
        """
        try:
            dict = self.create_transistor()
            transistor = dict["transistor"]

            existing_transistor_list = self.get_transistor_list()
            check = False
            for existing_transistor in existing_transistor_list:
                if existing_transistor == transistor.name:
                    check = True
                    self.InformationWindow = InformationWindow()
                    self.InformationWindow.run_information_window()
            if check == False:
                transistor.save()

                self.comboBox_export_transistor.addItem(transistor.name)
                self.comboBox_compare_transistor1.addItem(transistor.name)
                self.comboBox_compare_transistor2.addItem(transistor.name)
                self.comboBox_compare_transistor3.addItem(transistor.name)
                self.comboBox_topology_transistor1.addItem(transistor.name)
                self.comboBox_topology_transistor2.addItem(transistor.name)

                self.search_database_load_data()
                self.show_popup_message(f"<b>{transistor.name}</b> succsessfully created!")
                self.button_create_transistor_create.setDisabled(True)
        except:
            self.show_popup_message("Error: Transistor could not be created! Check if all inputs are correct!")


    def overwrite_transistor(self):
        """
        Overwrites a transistor stored in the local database

        :return: None
        """
        dict = self.create_transistor()
        transistor_new = dict["transistor"]

        transistor_old = tdb.load(transistor_new.name)

        id = transistor_old._id

        dict["transistor_args"].update({"_id": id})

        transistor_new = tdb.Transistor(dict["transistor_args"], dict["switch_args"], dict["diode_args"])

        # Add new DPT data to overwritten transistor object
        data_list = self.get_all_items_data_from_comboBox(self.comboBox_create_transistor_added_dpt)
        for data in data_list:
            try:
                transistor_new.add_dpt_measurement(data["new_dpt_dict"])
            except:
                pass

        transistor_new.save(overwrite=True)
        self.search_database_load_data()
        self.show_popup_message(f"Transistor <b>{transistor_new.name}</b> succsessfully overwritten!")
        self.button_create_transistor_create.setDisabled(True)

    def preview_transistor_on_virtual_datasheet(self):
        """
        Previews the transistor with paramaters given in the Create Transistor Tab on a virtual datasheet

        :return:
        """
        try:
            dict = self.create_transistor()
            transistor = dict["transistor"]

            html = transistor.export_datasheet(build_collection=True)
            with tempfile.NamedTemporaryFile('w', delete=False, suffix='.html') as f:
                url = 'file://' + f.name
                f.write(html)
            webbrowser.open(url)
            self.button_create_transistor_create.setDisabled(False)
        except:
            self.show_popup_message("Error: Transistor could not previewed! Check if all inputs are correct!")



    ###SEARCH DATABASE###

    def reset_filter_search_database(self):
        """
        Resets all filters in the search database tab

        :return: None
        """
        for widget in self.scrollAreaWidgetContents_search_database.children():
            if isinstance(widget, QLineEdit):
                widget.clear()

    def get_marked_transistor(self):
        """
        Get the marked transistor object from the search-database tab.

        :return: Transistor
        :rtype: Transistor
        """
        try:
            for i in range(self.tableWidget_search_database.columnCount()):
                if self.tableWidget_search_database.horizontalHeaderItem(i).text() == "NAME":
                    column = i
            selected_transistor_name = self.tableWidget_search_database.item(self.tableWidget_search_database.currentRow(), column).text()

            transistor = tdb.load(selected_transistor_name)
            return transistor

        except:
            self.show_popup_message("Error: No transistor selected!")


    def delete_marked_transistor_search_database_from_local_tdb(self):
        """
        Delete the marked transistor ('search transistor'-tab) from the local mongodb-database.

        """
        transistor = self.get_marked_transistor()
        transistor.delete()
        self.search_database_load_data()


    def load_from_search_database_into_create_transistor(self):
        """
        Loads a selected transistor from search database into create transistor

        :return: None
        """
        try:
            self.clear_create_transistor()


            for i in range(self.tableWidget_search_database.columnCount()):
                if self.tableWidget_search_database.horizontalHeaderItem(i).text() == "NAME":
                    column = i
            selected_transistor_name = self.tableWidget_search_database.item(self.tableWidget_search_database.currentRow(), column).text()

            transistor = tdb.load(selected_transistor_name)

            transistor_dict = transistor.convert_to_dict()
            transistor_switch_dict = transistor.switch.convert_to_dict()
            transistor_diode_dict = transistor.diode.convert_to_dict()

            self.lineEdit_create_transistor_name.setText(str(transistor_dict["name"]))
            self.comboBox_create_transistor_type.setCurrentText(str(transistor_dict["type"]))
            self.lineEdit_create_transistor_author.setText(str(transistor_dict["author"]))
            self.lineEdit_create_transistor_comment.setText(str(transistor_dict["comment"]))
            self.comboBox_create_transistor_manufacturer.setCurrentText(str(transistor_dict["manufacturer"]))
            self.lineEdit_create_transistor_datasheet_hyperlink.setText(str(transistor_dict["datasheet_hyperlink"]))
            self.lineEdit_create_transistor_datasheet_date.setText(str(transistor_dict["datasheet_date"]))
            self.lineEdit_create_transistor_datasheet_version.setText(str(transistor_dict["datasheet_version"]))
            self.lineEdit_create_transistor_housing_area.setText(str(transistor_dict["housing_area"]))
            self.lineEdit_create_transistor_cooling_area.setText(str(transistor_dict["cooling_area"]))
            self.comboBox_create_transistor_housing_type.setCurrentText(transistor_dict["housing_type"])
            self.lineEdit_create_transistor_v_abs_max.setText(str(transistor_dict["v_abs_max"]))
            self.lineEdit_create_transistor_i_abs_max.setText(str(transistor_dict["i_abs_max"]))
            self.lineEdit_create_transistor_i_cont.setText(str(transistor_dict["i_cont"]))

            if transistor_dict["c_oss_er"] is not None:
                self.lineEdit_create_transistor_transistor_c_oss_er_c_o.setText(str(transistor_dict["c_oss_er"]["c_o"]))
                self.lineEdit_create_transistor_transistor_c_oss_er_v_gs.setText(str(transistor_dict["c_oss_er"]["v_gs"]))
                self.lineEdit_create_transistor_transistor_c_oss_er_v_ds.setText(str(transistor_dict["c_oss_er"]["v_ds"]))

            if transistor_dict["c_oss_tr"] is not None:
                self.lineEdit_create_transistor_transistor_c_oss_tr_c_o.setText(str(transistor_dict["c_oss_tr"]["c_o"]))
                self.lineEdit_create_transistor_transistor_c_oss_tr_v_gs.setText(str(transistor_dict["c_oss_tr"]["v_gs"]))
                self.lineEdit_create_transistor_transistor_c_oss_tr_v_ds.setText(str(transistor_dict["c_oss_tr"]["v_ds"]))

            self.lineEdit_create_transistor_t_c_max.setText(str(transistor_dict["t_c_max"]))
            self.lineEdit_create_transistor_r_g_int.setText(str(transistor_dict["r_g_int"]))
            self.lineEdit_create_transistor_r_th_cs.setText(str(transistor_dict["r_th_cs"]))
            self.lineEdit_create_transistor_r_th_diode_cs.setText(str(transistor_dict["r_th_diode_cs"]))
            self.lineEdit_create_transistor_r_th_switch_cs.setText(str(transistor_dict["r_th_switch_cs"]))
            self.lineEdit_create_transistor_r_g_on_recommended.setText(str(transistor_dict["r_g_on_recommended"]))
            self.lineEdit_create_transistor_r_g_off_recommended.setText(str(transistor_dict["r_g_off_recommended"]))
            self.lineEdit_create_transistor_c_iss_fix.setText(str(transistor_dict["c_iss_fix"]))
            self.lineEdit_create_transistor_c_oss_fix.setText(str(transistor_dict["c_oss_fix"]))
            self.lineEdit_create_transistor_c_rss_fix.setText(str(transistor_dict["c_rss_fix"]))

            #SWITCH KEYS###
            self.comboBox_create_transistor_switch_manufacturer.setCurrentText(str(transistor_switch_dict["manufacturer"]))
            self.lineEdit_create_transistor_switch_technology.setText(str(transistor_switch_dict["technology"]))
            self.lineEdit_create_transistor_switch_comment.setText(str(transistor_switch_dict["comment"]))
            self.lineEdit_create_transistor_switch_t_j_max.setText(str(transistor_switch_dict["t_j_max"]))

            try:
                r_th_vector_list = transistor_switch_dict["thermal_foster"]["r_th_vector"]
                r_th_vector = ""
                for i in range(len(r_th_vector_list)):
                    r_th_vector = r_th_vector + str(r_th_vector_list[i]) + " "
                self.lineEdit_create_transistor_switch_r_th_vector.setText(r_th_vector)
            except:
                self.lineEdit_create_transistor_switch_r_th_vector.setText("None")

            self.lineEdit_create_transistor_switch_r_th_total.setText(str(transistor_switch_dict["thermal_foster"]["r_th_total"]))

            try:
                c_th_vector_list = transistor_switch_dict["thermal_foster"]["c_th_vector"]
                c_th_vector = ""
                for i in range(len(c_th_vector_list)):
                    c_th_vector = c_th_vector + str(c_th_vector_list[i]) + " "
                self.lineEdit_create_transistor_switch_c_th_vector.setText(c_th_vector)
            except:
                self.lineEdit_create_transistor_switch_c_th_vector.setText("None")

            self.lineEdit_create_transistor_switch_c_th_total.setText(str(transistor_switch_dict["thermal_foster"]["c_th_total"]))

            try:
                tau_vector_list = transistor_switch_dict["thermal_foster"]["tau_vector"]
                tau_vector = ""
                for i in range(len(tau_vector_list)):
                    tau_vector = tau_vector + str(tau_vector_list[i]) + " "
                self.lineEdit_create_transistor_switch_tau_vector.setText(tau_vector)
            except:
                self.lineEdit_create_transistor_switch_tau_vector.setText("None")

            self.lineEdit_create_transistor_switch_tau_total.setText(str(transistor_switch_dict["thermal_foster"]["tau_total"]))


            #DIODE KEYS#
            self.comboBox_create_transistor_diode_manufacturer.setCurrentText(str(transistor_diode_dict["manufacturer"]))
            self.lineEdit_create_transistor_diode_technology.setText(str(transistor_diode_dict["technology"]))
            self.lineEdit_create_transistor_diode_comment.setText(str(transistor_diode_dict["comment"]))
            self.lineEdit_create_transistor_diode_t_j_max.setText(str(transistor_diode_dict["t_j_max"]))

            try:
                r_th_vector_list = transistor_diode_dict["thermal_foster"]["r_th_vector"]
                r_th_vector = ""
                for i in range(len(r_th_vector_list)):
                    r_th_vector = r_th_vector + str(r_th_vector_list[i]) + " "
                self.lineEdit_create_transistor_diode_r_th_vector.setText(r_th_vector)
            except:
                self.lineEdit_create_transistor_diode_r_th_vector.setText("None")


            self.lineEdit_create_transistor_diode_r_th_total.setText(
                str(transistor_diode_dict["thermal_foster"]["r_th_total"]))


            try:
                c_th_vector_list = transistor_diode_dict["thermal_foster"]["c_th_vector"]
                c_th_vector = ""
                for i in range(len(c_th_vector_list)):
                    c_th_vector = c_th_vector + str(c_th_vector_list[i]) + " "
                self.lineEdit_create_transistor_diode_c_th_vector.setText(c_th_vector)
            except:
                self.lineEdit_create_transistor_diode_c_th_vector.setText("None")


            self.lineEdit_create_transistor_diode_c_th_total.setText(
                str(transistor_diode_dict["thermal_foster"]["c_th_total"]))


            try:
                tau_vector_list = transistor_diode_dict["thermal_foster"]["tau_vector"]
                tau_vector = ""
                for i in range(len(tau_vector_list)):
                    tau_vector = tau_vector + str(tau_vector_list[i]) + " "
                self.lineEdit_create_transistor_diode_tau_vector.setText(tau_vector)
            except:
                self.lineEdit_create_transistor_diode_tau_vector.setText("None")


            self.lineEdit_create_transistor_diode_tau_total.setText(
                str(transistor_diode_dict["thermal_foster"]["tau_total"]))


            #TRANSISTOR CURVES#
            graph_v_ecoss = transistor_dict["graph_v_ecoss"]
            if graph_v_ecoss is not None and graph_v_ecoss is not []:
                self.comboBox_create_transistor_added_curve_v_ecoss.addItem(f"V_Ecoss Curve", {"graph_v_ecoss": np.array(graph_v_ecoss)})



            list = transistor_dict["c_iss"]
            list = self.convert_graph_key_in_list_to_array(list)
            for i in range(len(list)):
                t_j = str(list[i]["t_j"])+" °C" if list[i]["t_j"] is not None else "None"
                if list[i] is not None and list[i] != []:
                    self.comboBox_create_transistor_added_c_iss_normal.addItem(f"C_iss Curve, T_j = {t_j}", list[i])



            list = transistor_dict["c_oss"]
            list = self.convert_graph_key_in_list_to_array(list)
            for i in range(len(list)):
                t_j = str(list[i]["t_j"])+" °C" if list[i]["t_j"] is not None else "None"
                if list[i] is not None and list[i] != []:
                    self.comboBox_create_transistor_added_c_oss_normal.addItem(f"C_oss Curve, T_j = {t_j}", list[i])


            list = transistor_dict["c_rss"]
            list = self.convert_graph_key_in_list_to_array(list)
            for i in range(len(list)):
                t_j = str(list[i]["t_j"])+" °C" if list[i]["t_j"] is not None else "None"
                if list[i] is not None and list[i] != []:
                    self.comboBox_create_transistor_added_c_rss_normal.addItem(f"C_rss Curve, T_j = {t_j}", list[i])

            #SWITCH CURVES#

            thermal_foster_switch = transistor_switch_dict["thermal_foster"]
            if thermal_foster_switch["graph_t_rthjc"] is not None and thermal_foster_switch["graph_t_rthjc"] is not []:
                self.comboBox_create_transistor_switch_added_curve_t_rthjc.addItem("T_Rthjc Curve", {"graph_t_rthjc": np.array(thermal_foster_switch["graph_t_rthjc"])})




            list = transistor_switch_dict["channel"]
            list = self.convert_graph_key_in_list_to_array(list)
            for i in range(len(list)):
                t_j = str(list[i]["t_j"])+" °C" if list[i]["t_j"] is not None else "None"
                v_g = str(list[i]["v_g"])+" V" if list[i]["v_g"] is not None else "None"
                if list[i] is not None and list[i] != []:
                    self.comboBox_create_transistor_switch_added_curves_channel_data.addItem(f"Switch Channel Curve, T_j = {t_j}, V_g = {v_g}", list[i])



            list = transistor_switch_dict["e_on"]
            list = self.convert_graph_key_in_list_to_array(list)
            for i in range(len(list)):
                list[i].update({"e_on_off": "e_on"})
                t_j = str(list[i]["t_j"])+" °C" if list[i]["t_j"] is not None else "None"
                v_g = str(list[i]["v_g"])+" V" if list[i]["v_g"] is not None else "None"
                r_g = str(list[i]["r_g"])+" Ω" if list[i]["r_g"] is not None else "None"
                v_supply = str(list[i]["v_supply"])+" V" if list[i]["v_supply"] is not None else "None"
                if list[i] is not None and list[i] != []:
                    self.comboBox_create_transistor_switch_added_curves_switching_losses.addItem(
                        f"E_on: T_j = {t_j}, V_g = {v_g}, R_g = {r_g}, V_supply = {v_supply}", list[i])

            list = transistor_switch_dict["e_off"]
            list = self.convert_graph_key_in_list_to_array(list)
            for i in range(len(list)):
                list[i].update({"e_on_off": "e_off"})
                t_j = str(list[i]["t_j"])+" °C" if list[i]["t_j"] is not None else "None"
                v_g = str(list[i]["v_g"])+" V" if list[i]["v_g"] is not None else "None"
                r_g = str(list[i]["r_g"])+" Ω" if list[i]["r_g"] is not None else "None"
                v_supply = str(list[i]["v_supply"])+"V" if list[i]["v_supply"] is not None else "None"
                if list[i] is not None and list[i] != []:
                    self.comboBox_create_transistor_switch_added_curves_switching_losses.addItem(
                        f"E_off: T_j = {t_j}, V_g = {v_g}, R_g = {r_g}, V_supply = {v_supply}", list[i])


            list = transistor_switch_dict["soa"]
            list = self.convert_graph_key_in_list_to_array(list)
            for i in range(len(list)):
                t_c = str(list[i]["t_c"])+" °C" if list[i]["t_c"] is not None else "None"
                time_pulse = str(list[i]["time_pulse"])+" s" if list[i]["time_pulse"] is not None else "None"
                if list[i] is not None and list[i] != []:
                    self.comboBox_create_transistor_switch_added_curves_soa_t_pulse.addItem(
                        f"T_c = {t_c}, Time_pulse = {time_pulse}", list[i])

            list = transistor_switch_dict["charge_curve"]
            list = self.convert_graph_key_in_list_to_array(list)
            for i in range(len(list)):
                i_channel = str(list[i]["i_channel"])+" A" if list[i]["i_channel"] is not None else "None"
                t_j = str(list[i]["t_j"])+" °C" if list[i]["t_j"] is not None else "None"
                v_supply = str(list[i]["v_supply"])+" V" if list[i]["v_supply"] is not None else "None"
                i_g = str(list[i]["i_g"])+" A" if list[i]["i_g"] is not None else "None"
                if list[i] is not None and list[i] != []:
                    self.comboBox_create_transistor_switch_added_curves_gate_charge.addItem(
                        f"I_channel = {i_channel}, T_j = {t_j}, V_supply = {v_supply}, I_g = {i_g}", list[i])

            list = transistor_switch_dict["r_channel_th"]
            list = self.convert_graph_key_in_list_to_array(list)
            for i in range(len(list)):
                i_channel = str(list[i]["i_channel"])+" A" if list[i]["i_channel"] is not None else "None"
                v_g = str(list[i]["v_g"])+" V" if list[i]["v_g"] is not None else "None"
                r_channel_nominal = str(list[i]["r_channel_nominal"])+" Ω" if list[i]["r_channel_nominal"] is not None else "None"
                if list[i] is not None and list[i] != []:
                    self.comboBox_create_transistor_switch_added_curves_r_on.addItem(
                        f"I_channel = {i_channel}, V_g = {v_g}, R_channel_nominal = {r_channel_nominal}", list[i])



            #DIODE CURVES#

            thermal_foster_diode = transistor_diode_dict["thermal_foster"]
            if thermal_foster_diode["graph_t_rthjc"] is not None and thermal_foster_diode["graph_t_rthjc"] is not []:
                self.comboBox_create_transistor_diode_added_curve_t_rthjc.addItem("T_Rthjc Curve", {"graph_t_rthjc": np.array(thermal_foster_diode["graph_t_rthjc"])})



            list = transistor_diode_dict["channel"]
            list = self.convert_graph_key_in_list_to_array(list)
            for i in range(len(list)):
                t_j = str(list[i]["t_j"])+" °C" if list[i]["t_j"] is not None else "None"
                v_g = str(list[i]["v_g"])+" V" if list[i]["v_g"] is not None else "None"
                if list[i] is not None and list[i] != []:
                    self.comboBox_create_transistor_diode_added_curves_channel_data.addItem(
                        f"Diode Channel Curve, T_j = {t_j}, V_g = {v_g}", list[i])

            list = transistor_diode_dict["e_rr"]
            list = self.convert_graph_key_in_list_to_array(list)
            for i in range(len(list)):
                t_j = str(list[i]["t_j"])+"° C" if list[i]["t_j"] is not None else "None"
                v_g = str(list[i]["v_g"])+" V" if list[i]["v_g"] is not None else "None"
                r_g = str(list[i]["r_g"])+" Ω" if list[i]["r_g"] is not None else "None"
                v_supply = str(list[i]["v_supply"])+" V" if list[i]["v_supply"] is not None else "None "
                if list[i] is not None and list[i] != []:
                    self.comboBox_create_transistor_diode_added_curves_switching_losses.addItem(
                        f"T_j = {t_j}, V_g = {v_g}, R_g = {r_g}, V_supply = {v_supply}", list[i])



            list = transistor_switch_dict["e_on_meas"]
            list = self.convert_graph_key_in_list_to_array(list)
            for i in range(len(list)):
                t_j = str(list[i]["t_j"]) + "° C" if list[i]["t_j"] is not None else "None"
                v_g = str(list[i]["v_g"]) + " V" if list[i]["v_g"] is not None else "None"
                v_g_off = str(list[i]["v_g_off"]) + " V" if list[i]["v_g_off"] is not None else "None"
                r_g = str(list[i]["r_g"]) + " Ω" if list[i]["r_g"] is not None else "None"
                v_supply = str(list[i]["v_supply"]) + " V" if list[i]["v_supply"] is not None else "None "
                load_inductance = str(list[i]["load_inductance"]) + " F" if list[i]["load_inductance"] is not None else "None "
                commutation_inductance = str(list[i]["commutation_inductance"]) + " F" if list[i]["commutation_inductance"] is not None else "None "
                measurement_date = str(list[i]["measurement_date"]) if list[i]["measurement_date"] is not None else "None"
                measurement_testbench = str(list[i]["measurement_testbench"]) if list[i]["measurement_testbench"] is not None else "None"
                commutation_device = str(list[i]["commutation_device"]) if list[i]["commutation_device"] is not None else "None"
                comment = str(list[i]["comment"]) if list[i]["comment"] is not None else "None"

                if list[i] is not None and list[i] != []:
                    self.comboBox_create_transistor_added_dpt.addItem(
                        f"E_on: T_j = {t_j}, V_g = {v_g}, V_g_off = {v_g_off}, R_g = {r_g}, V_supply = {v_supply}\n"
                        f"Load Inductance = {load_inductance}, Commutation Inductance = {commutation_inductance}\n"
                        f"Measurement Date: {measurement_date}\nMeasurement Testbench: {measurement_testbench}\n"
                        f"Commutation Device: {commutation_device}\nComment: {comment}", list[i])

            list = transistor_switch_dict["e_off_meas"]
            list = self.convert_graph_key_in_list_to_array(list)
            for i in range(len(list)):
                t_j = str(list[i]["t_j"]) + "° C" if list[i]["t_j"] is not None else "None"
                v_g = str(list[i]["v_g"]) + " V" if list[i]["v_g"] is not None else "None"
                v_g_off = str(list[i]["v_g_off"]) + " V" if list[i]["v_g_off"] is not None else "None"
                r_g = str(list[i]["r_g"]) + " Ω" if list[i]["r_g"] is not None else "None"
                v_supply = str(list[i]["v_supply"]) + " V" if list[i]["v_supply"] is not None else "None "
                load_inductance = str(list[i]["load_inductance"]) + " F" if list[i]["load_inductance"] is not None else "None "
                commutation_inductance = str(list[i]["commutation_inductance"]) + " F" if list[i]["commutation_inductance"] is not None else "None "
                measurement_date = str(list[i]["measurement_date"]) if list[i]["measurement_date"] is not None else "None"
                measurement_testbench = str(list[i]["measurement_testbench"]) if list[i]["measurement_testbench"] is not None else "None"
                commutation_device = str(list[i]["commutation_device"]) if list[i]["commutation_device"] is not None else "None"
                comment = str(list[i]["comment"]) if list[i]["comment"] is not None else "None"

                if list[i] is not None and list[i] != []:
                    self.comboBox_create_transistor_added_dpt.addItem(
                        f"E_off: T_j = {t_j}, V_g = {v_g}, V_g_off = {v_g_off}, R_g = {r_g}, V_supply = {v_supply}\n"
                        f"Load Inductance = {load_inductance}, Commutation Inductance = {commutation_inductance}\n"
                        f"Measurement Date: {measurement_date}\nMeasurement Testbench: {measurement_testbench}\n"
                        f"Commutation Device: {commutation_device}\nComment: {comment}", list[i])


            raw_measurement_data = transistor_dict["raw_measurement_data"]
            e_on_meas = transistor_switch_dict["e_on_meas"]
            e_off_meas = transistor_switch_dict["e_off_meas"]
            dpt_data = {"raw_measurement_data": raw_measurement_data,
                        "e_on_meas": e_on_meas,
                        "e_off_meas": e_off_meas}
            if raw_measurement_data is not None and raw_measurement_data != []:
                self.comboBox_create_transistor_added_dpt.addItem("All DPT Measurement Data", dpt_data)


            #Clear all lineEdits containing "None"
            for widget in self.scrollAreaWidgetContents_create_transistor.children():
                if isinstance(widget, QLineEdit):
                    if widget.text() == "None":
                        widget.clear()


            self.show_popup_message(f"<b>{transistor.name}</b> successfully loaded into Create Transistor!")
        except:
            self.show_popup_message("Error: No transistor selected!")

    def convert_graph_key_in_list_to_array(self, curve_list):
        """
        Converts the graph key in a list of curves to an numpy array so that the edited transistor can be loaded into the database

        :return: curve list with graph key converted to a numpy array
        """
        for i in range(len(curve_list)):
            for key in curve_list[i].keys():
                if "graph" in key and curve_list[i][key] is not None:
                    array = np.array(curve_list[i][key])
                    curve_list[i][key] = array
        return curve_list


    def load_from_search_database_into_exporting_tools(self):
        """
        Loads a selected transistor from search database into exporting tools

        :return: None
        """
        try:
            for i in range(self.tableWidget_search_database.columnCount()):
                if self.tableWidget_search_database.horizontalHeaderItem(i).text() == "NAME":
                    column = i
            selected_transistor_name = self.tableWidget_search_database.item(self.tableWidget_search_database.currentRow(), column).text()

            self.comboBox_export_transistor.setCurrentText(selected_transistor_name)
            self.show_popup_message(f"<b>{selected_transistor_name}</b> successfully loaded into Exporting Tools!")
        except:
            self.show_popup_message("Error: No transistor selected!")


    def load_from_search_database_into_comparison_tools(self):
        """
        Loads a selected transistor from search database into comparison tools

        :return: None
        """
        try:
            for i in range(self.tableWidget_search_database.columnCount()):
                if self.tableWidget_search_database.horizontalHeaderItem(i).text() == "NAME":
                    column = i
            selected_transistor_name = self.tableWidget_search_database.item(self.tableWidget_search_database.currentRow(), column).text()

            target_transistor = self.comboBox_search_database_load_comparison_tools.currentText()

            if target_transistor == "Transistor1":
                self.comboBox_compare_transistor1.setCurrentText(selected_transistor_name)

            if target_transistor == "Transistor2":
                self.comboBox_compare_transistor2.setCurrentText(selected_transistor_name)

            if target_transistor == "Transistor3":
                self.comboBox_compare_transistor3.setCurrentText(selected_transistor_name)

            self.show_popup_message(f"<b>{selected_transistor_name}</b> successfully loaded as {target_transistor} into Comparison Tools!")
        except:
            self.show_popup_message("Error: No transistor selected!")


    def load_from_search_database_into_topology_calculator(self):
        """
        Loads a selected transistor from search database into topology calculator

        :return: None
        """
        try:
            for i in range(self.tableWidget_search_database.columnCount()):
                if self.tableWidget_search_database.horizontalHeaderItem(i).text() == "NAME":
                    column = i
            selected_transistor_name = self.tableWidget_search_database.item(
                self.tableWidget_search_database.currentRow(), column).text()

            target_transistor = self.comboBox_search_database_load_topology_calculator.currentText()


            if target_transistor == "Transistor1":
                self.comboBox_topology_transistor1.setCurrentText(selected_transistor_name)

            if target_transistor == "Transistor2":
                self.comboBox_topology_transistor2.setCurrentText(selected_transistor_name)

            self.show_popup_message(f"<b>{selected_transistor_name}</b> successfully loaded as {target_transistor} into Topology Calculator!")

        except:
            self.show_popup_message("Error: No transistor selected!")


    def search_database_load_data(self):
        """
        Loads the data from the transistordatabase into the tableWidget while taking into account all the set filters

        :return: None
        """
        transistor_list = tdb.print_tdb()
        transistordatabase = []

        for i in range(len(transistor_list)):
            transistor = tdb.load(transistor_list[i])

            transistor_dict = transistor.convert_to_dict()
            switch_dict = transistor.switch.convert_to_dict()
            diode_dict = transistor.diode.convert_to_dict()

            switch_dict_keys = list(switch_dict.keys())
            for i in range(len(switch_dict_keys)):
                switch_dict_keys[i] = "switch_" + switch_dict_keys[i]

            switch_dict_keys_new = switch_dict_keys
            switch_dict_keys_old = list(switch_dict.keys())

            for i in range(len(switch_dict)):
                switch_dict[switch_dict_keys_new[i]] = switch_dict.pop(switch_dict_keys_old[i])


            diode_dict_keys = list(diode_dict.keys())
            for i in range(len(diode_dict_keys)):
                diode_dict_keys[i] = "diode_" + diode_dict_keys[i]

            diode_dict_keys_new = diode_dict_keys
            diode_dict_keys_old = list(diode_dict.keys())

            for i in range(len(diode_dict)):
                diode_dict[diode_dict_keys_new[i]] = diode_dict.pop(diode_dict_keys_old[i])
            transistor_dict = {**transistor_dict, **switch_dict, **diode_dict}

            transistordatabase.append(transistor_dict)

        transistordatabase_keys = list(transistor_dict.keys())


        keys_to_remove = ["c_oss", "c_iss", "c_rss", "raw_measurement_data", "graph_v_ecoss", "c_oss_tr",
                          "c_oss_er", "diode", "switch", "switch_channel", "switch_e_on",
                          "switch_e_off", "switch_e_on_meas", "switch_e_off_meas", "switch_linearized_switch",
                          "switch_r_channel_th", "switch_charge_curve", "switch_soa", "diode_thermal_foster", "switch_thermal_foster",
                          "diode_channel", "diode_e_rr", "diode_linearized_diode", "diode_soa"]


        if self.checkBox_search_database_id.isChecked() == False:
            keys_to_remove.append("_id")
        if self.checkBox_search_database_name.isChecked() == False:
            keys_to_remove.append("name")
        if self.checkBox_search_database_type.isChecked() == False:
            keys_to_remove.append("type")
        if self.checkBox_search_database_author.isChecked() == False:
            keys_to_remove.append("author")
        if self.checkBox_search_database_technology.isChecked() == False:
            keys_to_remove.append("technology")
        if self.checkBox_search_database_template_version.isChecked() == False:
            keys_to_remove.append("template_version")
        if self.checkBox_search_database_template_date.isChecked() == False:
            keys_to_remove.append("template_date")
        if self.checkBox_search_database_creation_date.isChecked() == False:
            keys_to_remove.append("creation_date")
        if self.checkBox_search_database_last_modified.isChecked() == False:
            keys_to_remove.append("last_modified")
        if self.checkBox_search_database_comment.isChecked() == False:
            keys_to_remove.append("comment")
        if self.checkBox_search_database_datasheet_hyperlink.isChecked() == False:
            keys_to_remove.append("datasheet_hyperlink")
        if self.checkBox_search_database_datasheet_date.isChecked() == False:
            keys_to_remove.append("datasheet_date")
        if self.checkBox_search_database_datasheet_version.isChecked() == False:
            keys_to_remove.append("datasheet_version")
        if self.checkBox_search_database_housing_area.isChecked() == False:
            keys_to_remove.append("housing_area")
        if self.checkBox_search_database_cooling_area.isChecked() == False:
            keys_to_remove.append("cooling_area")
        if self.checkBox_search_database_t_c_max.isChecked() == False:
            keys_to_remove.append("t_c_max")
        if self.checkBox_search_database_r_g_int.isChecked() == False:
            keys_to_remove.append("r_g_int")
        if self.checkBox_search_database_r_g_on_recommended.isChecked() == False:
            keys_to_remove.append("r_g_on_recommended")
        if self.checkBox_search_database_r_g_off_recommended.isChecked() == False:
            keys_to_remove.append("r_g_off_recommended")
        if self.checkBox_search_database_c_oss_fix.isChecked() == False:
            keys_to_remove.append("c_oss_fix")
        if self.checkBox_search_database_c_iss_fix.isChecked() == False:
            keys_to_remove.append("c_iss_fix")
        if self.checkBox_search_database_c_rss_fix.isChecked() == False:
            keys_to_remove.append("c_rss_fix")
        if self.checkBox_search_database_housing_type.isChecked() == False:
            keys_to_remove.append("housing_type")
        if self.checkBox_search_database_manufacturer.isChecked() == False:
            keys_to_remove.append("manufacturer")
        if self.checkBox_search_database_r_th_cs.isChecked() == False:
            keys_to_remove.append("r_th_cs")
        if self.checkBox_search_database_r_th_switch_cs.isChecked() == False:
            keys_to_remove.append("r_th_switch_cs")
        if self.checkBox_search_database_r_th_diode_cs.isChecked() == False:
            keys_to_remove.append("r_th_diode_cs")
        if self.checkBox_search_database_v_abs_max.isChecked() == False:
            keys_to_remove.append("v_abs_max")
        if self.checkBox_search_database_i_abs_max.isChecked() == False:
            keys_to_remove.append("i_abs_max")
        if self.checkBox_search_database_i_cont.isChecked() == False:
            keys_to_remove.append("i_cont")
        if self.checkBox_search_database_switch_t_j_max.isChecked() == False:
            keys_to_remove.append("switch_t_j_max")
        if self.checkBox_search_database_switch_comment.isChecked() == False:
            keys_to_remove.append("switch_comment")
        if self.checkBox_search_database_switch_manufacturer.isChecked() == False:
            keys_to_remove.append("switch_manufacturer")
        if self.checkBox_search_database_switch_technology.isChecked() == False:
            keys_to_remove.append("switch_technology")
        if self.checkBox_search_database_diode_comment.isChecked() == False:
            keys_to_remove.append("diode_comment")
        if self.checkBox_search_database_diode_manufacturer.isChecked() == False:
            keys_to_remove.append("diode_manufacturer")
        if self.checkBox_search_database_diode_technology.isChecked() == False:
            keys_to_remove.append("diode_technology")
        if self.checkBox_search_database_diode_t_j_max.isChecked() == False:
            keys_to_remove.append("diode_t_j_max")


        for key in keys_to_remove:
            transistordatabase_keys.remove(key)


        for transistor in transistordatabase:
            for key in transistordatabase_keys:
                if transistor[key] == None:
                    transistor[key] = 0


        if self.lineEdit_search_database_id.text() != "":
            id = self.lineEdit_search_database_id.text()
        else:
            id = ""
        if self.lineEdit_search_database_name.text() != "":
            name = self.lineEdit_search_database_name.text()
        else:
            name = ""
        if self.lineEdit_search_database_type.text() != "":
            type = self.lineEdit_search_database_type.text()
        else:
            type = ""

        if self.lineEdit_search_database_author.text() != "":
            author = self.lineEdit_search_database_author.text()
        else:
            author = ""
        if self.lineEdit_search_database_technology.text() != "":
            technology = self.lineEdit_search_database_technology.text()
        else:
            technology = ""
        if self.lineEdit_search_database_template_version.text() != "":
            template_version = self.lineEdit_search_database_template_version.text()
        else:
            template_version = ""
        if self.lineEdit_search_database_template_date.text() != "":
            template_date = self.lineEdit_search_database_template_date.text()
        else:
            template_date = ""
        if self.lineEdit_search_database_creation_date.text() != "":
            creation_date = self.lineEdit_search_database_creation_date.text()
        else:
            creation_date = ""
        if self.lineEdit_search_database_last_modified.text() != "":
            last_modified = self.lineEdit_search_database_last_modified.text()
        else:
            last_modified = ""
        if self.lineEdit_search_database_comment.text() != "":
            comment = self.lineEdit_search_database_comment.text()
        else:
            comment = ""
        if self.lineEdit_search_database_datasheet_hyperlink.text() != "":
            datasheet_hyperlink = self.lineEdit_search_database_datasheet_hyperlink.text()
        else:
            datasheet_hyperlink = ""
        if self.lineEdit_search_database_datasheet_date.text() != "":
            datasheet_date = self.lineEdit_search_database_datasheet_date.text()
        else:
            datasheet_date = ""
        if self.lineEdit_search_database_datasheet_version.text() != "":
            datasheet_version = self.lineEdit_search_database_datasheet_version.text()
        else:
            datasheet_version = ""
        if self.lineEdit_search_database_housing_type.text() != "":
            housing_type = self.lineEdit_search_database_housing_type.text()
        else:
            housing_type = ""
        if self.lineEdit_search_database_manufacturer.text() != "":
            manufacturer = self.lineEdit_search_database_manufacturer.text()
        else:
            manufacturer = ""
        if self.lineEdit_search_database_switch_comment.text() != "":
            switch_comment = self.lineEdit_search_database_switch_comment.text()
        else:
            switch_comment = ""
        if self.lineEdit_search_database_switch_manufacturer.text() != "":
            switch_manufacturer = self.lineEdit_search_database_switch_manufacturer.text()
        else:
            switch_manufacturer = ""
        if self.lineEdit_search_database_switch_technology.text() != "":
            switch_technology = self.lineEdit_search_database_switch_technology.text()
        else:
            switch_technology = ""
        if self.lineEdit_search_database_diode_comment.text() != "":
            diode_comment = self.lineEdit_search_database_diode_comment.text()
        else:
            diode_comment = ""
        if self.lineEdit_search_database_diode_manufacturer.text() != "":
            diode_manufacturer = self.lineEdit_search_database_diode_manufacturer.text()
        else:
            diode_manufacturer = ""
        if self.lineEdit_search_database_diode_technology.text() != "":
            diode_technology = self.lineEdit_search_database_diode_technology.text()
        else:
            diode_technology = ""
        if self.lineEdit_search_database_t_c_max_min.text() != "":
            t_c_max_min = self.lineEdit_search_database_t_c_max_min.text()
        else:
            t_c_max_min = -100000
        if self.lineEdit_search_database_t_c_max_max.text() != "":
            t_c_max_max = self.lineEdit_search_database_t_c_max_max.text()
        else:
            t_c_max_max = 100000
        if self.lineEdit_search_database_r_g_int_min.text() != "":
            r_g_int_min = self.lineEdit_search_database_r_g_int_min.text()
        else:
            r_g_int_min = -100000
        if self.lineEdit_search_database_r_g_int_max.text() != "":
            r_g_int_max = self.lineEdit_search_database_r_g_int_max.text()
        else:
            r_g_int_max = 100000
        if self.lineEdit_search_database_r_g_on_recommended_min.text() != "":
            r_g_on_recommended_min = self.lineEdit_search_database_r_g_on_recommended_min.text()
        else:
            r_g_on_recommended_min = -100000
        if self.lineEdit_search_database_r_g_on_recommended_max.text() != "":
            r_g_on_recommended_max = self.lineEdit_search_database_r_g_on_recommended_max.text()
        else:
            r_g_on_recommended_max = 100000
        if self.lineEdit_search_database_r_g_off_recommended_min.text() != "":
            r_g_off_recommended_min = self.lineEdit_search_database_r_g_off_recommended_min.text()
        else:
            r_g_off_recommended_min = -100000
        if self.lineEdit_search_database_r_g_off_recommended_max.text() != "":
            r_g_off_recommended_max = self.lineEdit_search_database_r_g_off_recommended_max.text()
        else:
            r_g_off_recommended_max = 100000
        if self.lineEdit_search_database_c_oss_fix_min.text() != "":
            c_oss_fix_min = self.lineEdit_search_database_c_oss_fix_min.text()
        else:
            c_oss_fix_min = -100000
        if self.lineEdit_search_database_c_oss_fix_max.text() != "":
            c_oss_fix_max = self.lineEdit_search_database_c_oss_fix_max.text()
        else:
            c_oss_fix_max = 100000
        if self.lineEdit_search_database_c_iss_fix_min.text() != "":
            c_iss_fix_min = self.lineEdit_search_database_c_iss_fix_min.text()
        else:
            c_iss_fix_min = -100000
        if self.lineEdit_search_database_c_iss_fix_max.text() != "":
            c_iss_fix_max = self.lineEdit_search_database_c_iss_fix_max.text()
        else:
            c_iss_fix_max = 100000
        if self.lineEdit_search_database_c_rss_fix_min.text() != "":
            c_rss_fix_min = self.lineEdit_search_database_c_rss_fix_min.text()
        else:
            c_rss_fix_min = -100000
        if self.lineEdit_search_database_c_rss_fix_max.text() != "":
            c_rss_fix_max = self.lineEdit_search_database_c_rss_fix_max.text()
        else:
            c_rss_fix_max = 100000
        if self.lineEdit_search_database_r_th_cs_min.text() != "":
            r_th_cs_min = self.lineEdit_search_database_r_th_cs_min.text()
        else:
            r_th_cs_min = -100000
        if self.lineEdit_search_database_r_th_cs_max.text() != "":
            r_th_cs_max = self.lineEdit_search_database_r_th_cs_max.text()
        else:
            r_th_cs_max = 100000
        if self.lineEdit_search_database_r_th_switch_cs_min.text() != "":
            r_th_switch_cs_min = self.lineEdit_search_database_r_th_switch_cs_min.text()
        else:
            r_th_switch_cs_min = -100000
        if self.lineEdit_search_database_r_th_switch_cs_max.text() != "":
            r_th_switch_cs_max = self.lineEdit_search_database_r_th_switch_cs_max.text()
        else:
            r_th_switch_cs_max = 100000

        if self.lineEdit_search_database_r_th_diode_cs_min.text() != "":
            r_th_diode_cs_min = self.lineEdit_search_database_r_th_diode_cs_min.text()
        else:
            r_th_diode_cs_min = -100000
        if self.lineEdit_search_database_r_th_diode_cs_max.text() != "":
            r_th_diode_cs_max = self.lineEdit_search_database_r_th_diode_cs_max.text()
        else:
            r_th_diode_cs_max = 100000

        if self.lineEdit_search_database_v_abs_max_min.text() != "":
            v_abs_max_min = self.lineEdit_search_database_v_abs_max_min.text()
        else:
            v_abs_max_min = -100000
        if self.lineEdit_search_database_v_abs_max_max.text() != "":
            v_abs_max_max = self.lineEdit_search_database_v_abs_max_max.text()
        else:
            v_abs_max_max = 100000

        if self.lineEdit_search_database_i_abs_max_min.text() != "":
            i_abs_max_min = self.lineEdit_search_database_i_abs_max_min.text()
        else:
            i_abs_max_min = -100000
        if self.lineEdit_search_database_i_abs_max_max.text() != "":
            i_abs_max_max = self.lineEdit_search_database_i_abs_max_max.text()
        else:
            i_abs_max_max = 100000
        if self.lineEdit_search_database_i_cont_min.text() != "":
            i_cont_min = self.lineEdit_search_database_i_cont_min.text()
        else:
            i_cont_min = -100000
        if self.lineEdit_search_database_i_cont_max.text() != "":
            i_cont_max = self.lineEdit_search_database_i_cont_max.text()
        else:
            i_cont_max = 100000
        if self.lineEdit_search_database_switch_t_j_max_min.text() != "":
            switch_t_j_max_min = self.lineEdit_search_database_switch_t_j_max_min.text()
        else:
            switch_t_j_max_min = -100000
        if self.lineEdit_search_database_switch_t_j_max_max.text() != "":
            switch_t_j_max_max = self.lineEdit_search_database_switch_t_j_max_max.text()
        else:
            switch_t_j_max_max = 100000
        if self.lineEdit_search_database_diode_t_j_max_min.text() != "":
            diode_t_j_max_min = self.lineEdit_search_database_diode_t_j_max_min.text()
        else:
            diode_t_j_max_min = -100000
        if self.lineEdit_search_database_diode_t_j_max_max.text() != "":
            diode_t_j_max_max = self.lineEdit_search_database_diode_t_j_max_max.text()
        else:
            diode_t_j_max_max = 100000
        if self.lineEdit_search_database_housing_area_min.text() != "":
            housing_area_min = self.lineEdit_search_database_housing_area_min.text()
        else:
            housing_area_min = -100000
        if self.lineEdit_search_database_housing_area_max.text() != "":
            housing_area_max = self.lineEdit_search_database_housing_area_max.text()
        else:
            housing_area_max = 100000
        if self.lineEdit_search_database_cooling_area_min.text() != "":
            cooling_area_min = self.lineEdit_search_database_cooling_area_min.text()
        else:
            cooling_area_min = -100000
        if self.lineEdit_search_database_cooling_area_max.text() != "":
            cooling_area_max = self.lineEdit_search_database_cooling_area_max.text()
        else:
            cooling_area_max = 100000

        try:
            transistordatabase_filtered = []
            for i in range(len(transistordatabase)):
                if type.lower() in str(transistordatabase[i]["type"]).lower() and \
                   name.lower() in str(transistordatabase[i]["name"]).lower() and \
                   id.lower() in str(transistordatabase[i]["_id"]).lower() and \
                   author.lower() in str(transistordatabase[i]["author"]).lower() and \
                   technology.lower() in str(transistordatabase[i]["technology"]).lower() and \
                   template_version.lower() in str(transistordatabase[i]["template_version"]).lower() and \
                   template_date.lower() in str(transistordatabase[i]["template_date"]).lower() and \
                   creation_date.lower() in str(transistordatabase[i]["creation_date"]).lower() and \
                   last_modified.lower() in str(transistordatabase[i]["last_modified"]).lower() and \
                   comment.lower() in str(transistordatabase[i]["comment"]).lower() and \
                   datasheet_hyperlink.lower() in str(transistordatabase[i]["datasheet_hyperlink"]).lower() and \
                   datasheet_date.lower() in str(transistordatabase[i]["datasheet_date"]).lower() and \
                   datasheet_version.lower() in str(transistordatabase[i]["datasheet_version"]).lower() and \
                   housing_type.lower() in str(transistordatabase[i]["housing_type"]).lower() and \
                   manufacturer.lower() in str(transistordatabase[i]["manufacturer"]).lower() and \
                   switch_comment.lower() in str(transistordatabase[i]["switch_comment"]).lower() and \
                   switch_manufacturer.lower() in str(transistordatabase[i]["switch_manufacturer"]).lower() and \
                   switch_technology.lower() in str(transistordatabase[i]["switch_technology"]).lower() and \
                   diode_comment.lower() in str(transistordatabase[i]["diode_comment"]).lower() and \
                   diode_manufacturer.lower() in str(transistordatabase[i]["diode_manufacturer"]).lower() and \
                   diode_technology.lower() in str(transistordatabase[i]["diode_technology"]).lower() and \
                   float(i_abs_max_min) < float(transistordatabase[i]["i_abs_max"]) < float(i_abs_max_max) and \
                   float(t_c_max_min) < float(transistordatabase[i]["t_c_max"]) < float(t_c_max_max) and \
                   float(r_g_int_min) < float(transistordatabase[i]["r_g_int"]) < float(r_g_int_max) and \
                   float(r_g_on_recommended_min) < float(transistordatabase[i]["r_g_on_recommended"]) < float(r_g_on_recommended_max) and \
                   float(r_g_off_recommended_min) < float(transistordatabase[i]["r_g_off_recommended"]) < float(r_g_off_recommended_max) and \
                   float(c_oss_fix_min) < float(transistordatabase[i]["c_oss_fix"]) < float(c_oss_fix_max) and \
                   float(c_iss_fix_min) < float(transistordatabase[i]["c_iss_fix"]) < float(c_iss_fix_max) and \
                   float(c_rss_fix_min) < float(transistordatabase[i]["c_rss_fix"]) < float(c_rss_fix_max) and \
                   float(r_th_cs_min) < float(transistordatabase[i]["r_th_cs"]) < float(r_th_cs_max) and \
                   float(r_th_switch_cs_min) < float(transistordatabase[i]["r_th_switch_cs"]) < float(r_th_switch_cs_max) and \
                   float(r_th_diode_cs_min) < float(transistordatabase[i]["r_th_diode_cs"]) < float(r_th_diode_cs_max) and \
                   float(v_abs_max_min) < float(transistordatabase[i]["v_abs_max"]) < float(v_abs_max_max) and \
                   float(i_cont_min) < float(transistordatabase[i]["i_cont"]) < float(i_cont_max) and \
                   float(switch_t_j_max_min) < float(transistordatabase[i]["switch_t_j_max"]) < float(switch_t_j_max_max) and \
                   float(diode_t_j_max_min) < float(transistordatabase[i]["diode_t_j_max"]) < float(diode_t_j_max_max) and \
                   float(housing_area_min) < float(transistordatabase[i]["housing_area"]) < float(housing_area_max) and \
                   float(cooling_area_min) < float(transistordatabase[i]["cooling_area"]) < float(cooling_area_max):
                    transistordatabase_filtered.append(transistordatabase[i])

            for transistor in transistordatabase_filtered:
                for key in keys_to_remove:
                    del transistor[key]

                for key in transistordatabase_keys:
                    if transistor[key] == 0:
                        transistor[key] = ""

            transistordatabase_keys_upper = []
            unit = ""
            for key in transistordatabase_keys:
                if key == "t_c_max" or key == "switch_t_j_max" or key == "diode_t_j_max":
                    unit = " [°C]"
                if key == "housing_area" or key == "cooling_area":
                    unit = " [m²]"
                if key == "r_g_int" or key == "r_g_on_recommended" or key == "r_g_off_recommended" or key == "r_th_cs" or key == "r_th_switch_cs" or key == "r_th_diode_cs":
                    unit = " [Ω]"
                if key == "c_oss_fix" or key == "c_iss_fix" or key == "c_rss_fix":
                    unit = " [F]"
                if key == "v_abs_max":
                    unit = " [V]"
                if key == "i_abs_max" or key == "i_cont":
                    unit = " [A]"
                transistordatabase_keys_upper.append(str(transistordatabase_keys[transistordatabase_keys.index(key)].upper()) + str(unit))
                unit = ""


            row = 0
            column = 0
            self.tableWidget_search_database.setRowCount(len(transistordatabase_filtered))
            self.tableWidget_search_database.setColumnCount(len(transistordatabase_keys))
            self.tableWidget_search_database.setHorizontalHeaderLabels(transistordatabase_keys_upper)
            for transistor in transistordatabase_filtered:
                for key in transistordatabase_keys:
                    item = QtWidgets.QTableWidgetItem()
                    try:
                        item.setData(QtCore.Qt.DisplayRole, float(transistor[key]))
                    except:
                        item.setData(QtCore.Qt.DisplayRole, str(transistor[key]))
                    self.tableWidget_search_database.setItem(row, column, item)
                    column = column + 1
        except:
            pass



    ###Exporting Tools###

    def export_datasheet(self):
        """
        Exports the virtual datasheet in form of pdf in the current working directory

        :return: pdf file is created in the current working directory
        """
        transistor = tdb.load(self.comboBox_export_transistor.currentText())
        transistor.export_datasheet()

        self.show_popup_message(f"Exported a virtual datasheet for <b>{transistor.name}</b> to <a href={pathlib.Path.cwd().as_uri()}>{pathlib.Path.cwd().as_uri()}</a>")




    def export_json(self):
        """
        Exports a json file in the current working directory

        :return: json file is created in the current working directory
        """
        transistor = tdb.load(self.comboBox_export_transistor.currentText())
        transistor.export_json()

        self.show_popup_message(
            f"Exported a json file for <b>{transistor.name}</b> to <a href={pathlib.Path.cwd().as_uri()}>{pathlib.Path.cwd().as_uri()}</a>")


    def export_matlab(self):
        """
        Exports a matlab file in the current working directory

        :return: matlab file is created in the current working directory
        """
        try:
            transistor = tdb.load(self.comboBox_export_transistor.currentText())
            transistor_parallel = transistor.parallel_transistors(int(self.lineEdit_export_number_parallel_transistors.text()))
            transistor_parallel.export_matlab()

            self.show_popup_message(
                f"Exported a MATLAB file for {transistor_parallel.name} to <a href={pathlib.Path.cwd().as_uri()}>{pathlib.Path.cwd().as_uri()}</a>")
        except:
            self.show_popup_message("Invalid input for number of parallel transistors!")


    def export_simulink(self):
        """
        Exports a simulink file in the current working directory

        :return: simulink file is created in the current working directory
        """
        try:
            transistor = tdb.load(self.comboBox_export_transistor.currentText())
            if transistor.type == "IGBT":
                transistor_parallel = transistor.parallel_transistors(int(self.lineEdit_export_number_parallel_transistors.text()))
                transistor_parallel.export_simulink_loss_model(r_g_on=float(self.lineEdit_export_simulink_r_g_on.text()),
                                                               r_g_off=float(self.lineEdit_export_simulink_r_g_off.text()),
                                                               v_supply=float(self.lineEdit_export_simulink_v_supply.text()),
                                                               normalize_t_to_v=float(self.lineEdit_export_simulink_normalize_t_to_v.text()))

                self.show_popup_message(
                    f"Exported a Simulink file for {transistor_parallel.name} to <a href={pathlib.Path.cwd().as_uri()}>{pathlib.Path.cwd().as_uri()}</a>")
            else:
                self.show_popup_message("Error: Exporting simulink files is working for IGBTs only!")

        except:
            self.show_popup_message("Error: One or more invalid inputs!")


    def export_plecs(self):
        """
        Exports a PLECS file in the current working directory

        :return: PLECS file is created in the current working directory
        """
        transistor = tdb.load(self.comboBox_export_transistor.currentText())

        transistor_parallel = transistor.parallel_transistors(
            int(self.lineEdit_export_number_parallel_transistors.text()))

        try:
            gate_voltages = [float(self.lineEdit_export_plecs_v_g_on.text()),
                             float(self.lineEdit_export_plecs_v_g_off.text()),
                             float(self.lineEdit_export_plecs_v_d_on.text()),
                             float(self.lineEdit_export_plecs_v_d_off.text())]
            transistor_parallel.export_plecs(gate_voltages)
        except:
            transistor_parallel.export_plecs()

        self.show_popup_message(
            f"Exported a PLECS file for {transistor_parallel.name} to <a href={pathlib.Path.cwd().as_uri()}>{pathlib.Path.cwd().as_uri()}</a>")


    def export_gecko(self):
        """
        Exports GeckoCircuits files in the current working directory

        :return: GeckoCircuits files are created in the current working directory
        """
        try:
            transistor = tdb.load(self.comboBox_export_transistor.currentText())
            transistor_parallel = transistor.parallel_transistors(int(self.lineEdit_export_number_parallel_transistors.text()))
            transistor_parallel.export_geckocircuits(v_supply=float(self.lineEdit_export_gecko_v_supply.text()),
                                                     r_g_on=float(self.lineEdit_export_gecko_r_g_on.text()),
                                                     r_g_off=float(self.lineEdit_export_gecko_r_g_off.text()),
                                                     v_g_on=float(self.lineEdit_export_gecko_v_g_on.text()),
                                                     v_g_off=float(self.lineEdit_export_gecko_v_g_off.text()))


            self.show_popup_message(f"Exported GeckoCircuits files for {transistor_parallel.name} to <a href={pathlib.Path.cwd().as_uri()}>{pathlib.Path.cwd().as_uri()}</a>")
        except:
            self.show_popup_message("Error: One or more invalid inputs!")






    ###COMPARISON TOOLS###

    def compare_create_plot(self, widget_plot: QWidget, matplotlibwidget, comboBox_plot):
        """
        Adds a Matplotlib figure to a QWidget and creates a plot based on all the possible inputs and selections


        :param widget_compare_plot: widget for the matplotlib figure
        :type widget_compare_plot: QWidget
        :param matplotlibwidget_compare: matplotlib figure
        :param comboBox_compare_plot: comboBox to choose plot
        :return: None
        """

        matplotlibwidget.axis.clear()
        self.layout = QVBoxLayout(widget_plot)
        self.layout.addWidget(matplotlibwidget)

        try:
            matplotlibwidget.axis_cm.remove()
        except:
            pass

        try:
            transistor1 = tdb.load(self.comboBox_compare_transistor1.currentText())
            transistor2 = tdb.load(self.comboBox_compare_transistor2.currentText())
            transistor3 = tdb.load(self.comboBox_compare_transistor3.currentText())

            if self.lineEdit_compare_number_parallel_transistor1.text() != "1":
                transistor1 = transistor1.parallel_transistors(
                    int(self.lineEdit_compare_number_parallel_transistor1.text()))
            if self.lineEdit_compare_number_parallel_transistor2.text() != "1":
                transistor2 = transistor2.parallel_transistors(
                    int(self.lineEdit_compare_number_parallel_transistor2.text()))
            if self.lineEdit_compare_number_parallel_transistor3.text() != "1":
                transistor3 = transistor3.parallel_transistors(
                    int(self.lineEdit_compare_number_parallel_transistor3.text()))

            r_g_on1 = float(self.label_compare_r_g_on_value_transistor1.text())
            r_g_on2 = float(self.label_compare_r_g_on_value_transistor2.text())
            r_g_on3 = float(self.label_compare_r_g_on_value_transistor3.text())
            r_g_off1 = float(self.label_compare_r_g_off_value_transistor1.text())
            r_g_off2 = float(self.label_compare_r_g_off_value_transistor2.text())
            r_g_off3 = float(self.label_compare_r_g_off_value_transistor3.text())
            v_supply1 = float(self.lineEdit_compare_v_supply_transistor1.text())
            v_supply2 = float(self.lineEdit_compare_v_supply_transistor2.text())
            v_supply3 = float(self.lineEdit_compare_v_supply_transistor3.text())
            t_j1 = float(self.lineEdit_compare_t_j_transistor1.text())
            t_j2 = float(self.lineEdit_compare_t_j_transistor2.text())
            t_j3 = float(self.lineEdit_compare_t_j_transistor3.text())
            v_g_on1 = float(self.comboBox_compare_v_g_on_transistor1.currentText())
            v_g_on2 = float(self.comboBox_compare_v_g_on_transistor2.currentText())
            v_g_on3 = float(self.comboBox_compare_v_g_on_transistor3.currentText())

            if self.comboBox_compare_v_g_off_transistor1.count() >= 1:
                v_g_off1 = float(self.comboBox_compare_v_g_off_transistor1.currentText())
            else:
                v_g_off1 = None

            if self.comboBox_compare_v_g_off_transistor2.count() >= 1:
                v_g_off2 = float(self.comboBox_compare_v_g_off_transistor2.currentText())
            else:
                v_g_off2 = None

            if self.comboBox_compare_v_g_off_transistor3.count() >= 1:
                v_g_off3 = float(self.comboBox_compare_v_g_off_transistor3.currentText())
            else:
                v_g_off3 = None

            if comboBox_plot.currentText() == "Switch Energy Data vs. Channel Current Transistor1":
                comparison_tools_functions.plot_all_energy_data(transistor1, matplotlibwidget, "switch")
            if comboBox_plot.currentText() == "Switch Energy Data vs. Channel Current Transistor2":
                comparison_tools_functions.plot_all_energy_data(transistor2, matplotlibwidget, "switch")
            if comboBox_plot.currentText() == "Switch Energy Data vs. Channel Current Transistor3":
                comparison_tools_functions.plot_all_energy_data(transistor3, matplotlibwidget, "switch")
            if comboBox_plot.currentText() == "Diode Energy Data vs. Channel Current Transistor1":
                comparison_tools_functions.plot_all_energy_data(transistor1, matplotlibwidget, "diode")
            if comboBox_plot.currentText() == "Diode Energy Data vs. Channel Current Transistor2":
                comparison_tools_functions.plot_all_energy_data(transistor2, matplotlibwidget, "diode")
            if comboBox_plot.currentText() == "Diode Energy Data vs. Channel Current Transistor3":
                comparison_tools_functions.plot_all_energy_data(transistor3, matplotlibwidget, "diode")
            if comboBox_plot.currentText() == "Switch Energy Data vs. Gate Resistor Transistor1":
                comparison_tools_functions.plot_all_energy_data_r_g(transistor1, matplotlibwidget, "switch")
            if comboBox_plot.currentText() == "Switch Energy Data vs. Gate Resistor Transistor2":
                comparison_tools_functions.plot_all_energy_data_r_g(transistor2, matplotlibwidget, "switch")
            if comboBox_plot.currentText() == "Switch Energy Data vs. Gate Resistor Transistor3":
                comparison_tools_functions.plot_all_energy_data_r_g(transistor3, matplotlibwidget, "switch")
            if comboBox_plot.currentText() == "Diode Energy Data vs. Gate Resistor Transistor1":
                comparison_tools_functions.plot_all_energy_data_r_g(transistor1, matplotlibwidget, "diode")
            if comboBox_plot.currentText() == "Diode Energy Data vs. Gate Resistor Transistor2":
                comparison_tools_functions.plot_all_energy_data_r_g(transistor2, matplotlibwidget, "diode")
            if comboBox_plot.currentText() == "Diode Energy Data vs. Gate Resistor Transistor3":
                comparison_tools_functions.plot_all_energy_data_r_g(transistor3, matplotlibwidget, "diode")
            if comboBox_plot.currentText() == "Switch Channel Data Transistor1":
                comparison_tools_functions.plot_all_channel_data(transistor1, matplotlibwidget, "switch")
            if comboBox_plot.currentText() == "Switch Channel Data Transistor2":
                comparison_tools_functions.plot_all_channel_data(transistor2, matplotlibwidget, "switch")
            if comboBox_plot.currentText() == "Switch Channel Data Transistor3":
                comparison_tools_functions.plot_all_channel_data(transistor3, matplotlibwidget, "switch")
            if comboBox_plot.currentText() == "Diode Channel Data Transistor1":
                comparison_tools_functions.plot_all_channel_data(transistor1, matplotlibwidget, "diode")
            if comboBox_plot.currentText() == "Diode Channel Data Transistor2":
                comparison_tools_functions.plot_all_channel_data(transistor2, matplotlibwidget, "diode")
            if comboBox_plot.currentText() == "Diode Channel Data Transistor3":
                comparison_tools_functions.plot_all_channel_data(transistor3, matplotlibwidget, "diode")

            if comboBox_plot.currentText() == "Switch Turn-on Losses":
                comparison_tools_functions.plot_e_on(transistor1=transistor1,
                                                     transistor2=transistor2,
                                                     transistor3=transistor3,
                                                     matplotlibwidget=matplotlibwidget,
                                                     t_j1=t_j1,
                                                     t_j2=t_j2,
                                                     t_j3=t_j3,
                                                     r_g_on1=r_g_on1,
                                                     r_g_on2=r_g_on2,
                                                     r_g_on3=r_g_on3,
                                                     v_supply1=v_supply1,
                                                     v_supply2=v_supply2,
                                                     v_supply3=v_supply3)

            if comboBox_plot.currentText() == "Switch Turn-off Losses":
                comparison_tools_functions.plot_e_off(transistor1=transistor1,
                                                      transistor2=transistor2,
                                                      transistor3=transistor3,
                                                      matplotlibwidget=matplotlibwidget,
                                                      t_j1=t_j1,
                                                      t_j2=t_j2,
                                                      t_j3=t_j3,
                                                      r_g_off1=r_g_off1,
                                                      r_g_off2=r_g_off2,
                                                      r_g_off3=r_g_off3,
                                                      v_supply1=v_supply1,
                                                      v_supply2=v_supply2,
                                                      v_supply3=v_supply3)

            if comboBox_plot.currentText() == "Diode Reverse Recovery Losses":
                comparison_tools_functions.plot_e_rr(transistor1=transistor1,
                                                     transistor2=transistor2,
                                                     transistor3=transistor3,
                                                     matplotlibwidget=matplotlibwidget,
                                                     t_j1=t_j1,
                                                     t_j2=t_j2,
                                                     t_j3=t_j3,
                                                     r_g_off1=r_g_off1,
                                                     r_g_off2=r_g_off2,
                                                     r_g_off3=r_g_off3,
                                                     v_supply1=v_supply1,
                                                     v_supply2=v_supply2,
                                                     v_supply3=v_supply3)

            if comboBox_plot.currentText() == "Switch Channel Data":
                comparison_tools_functions.plot_channel(transistor1=transistor1,
                                                        transistor2=transistor2,
                                                        transistor3=transistor3,
                                                        matplotlibwidget=matplotlibwidget,
                                                        t_j1=t_j1,
                                                        t_j2=t_j2,
                                                        t_j3=t_j3,
                                                        v_g_on1=v_g_on1,
                                                        v_g_on2=v_g_on2,
                                                        v_g_on3=v_g_on3,
                                                        v_g_off1=v_g_off1,
                                                        v_g_off2=v_g_off2,
                                                        v_g_off3=v_g_off3,
                                                        switch_diode="switch")

            if comboBox_plot.currentText() == "Diode Channel Data":
                comparison_tools_functions.plot_channel(transistor1=transistor1,
                                                        transistor2=transistor2,
                                                        transistor3=transistor3,
                                                        matplotlibwidget=matplotlibwidget,
                                                        t_j1=t_j1,
                                                        t_j2=t_j2,
                                                        t_j3=t_j3,
                                                        v_g_on1=v_g_on1,
                                                        v_g_on2=v_g_on2,
                                                        v_g_on3=v_g_on3,
                                                        v_g_off1=v_g_off1,
                                                        v_g_off2=v_g_off2,
                                                        v_g_off3=v_g_off3,
                                                        switch_diode="diode")

            if comboBox_plot.currentText() == "Output Capacitance Charge vs. Channel Voltage":
                comparison_tools_functions.plot_v_qoss(transistor1=transistor1,
                                                       transistor2=transistor2,
                                                       transistor3=transistor3,
                                                       matplotlibwidget=matplotlibwidget)

            if comboBox_plot.currentText() == "Output Capacitance Energy vs. Channel Voltage":
                comparison_tools_functions.plot_v_eoss(transistor1=transistor1,
                                                       transistor2=transistor2,
                                                       transistor3=transistor3,
                                                       matplotlibwidget=matplotlibwidget)
        except:
            self.show_popup_message("Error: Inputs are missing or not numeric!")


    def compare_update_plots(self):
        """
        Runs the function "topology_create_plot" for all QWidgets,
        Matplotlib figures and ComboBoxes

        :return: None
        """
        self.compare_create_plot(self.widget_compare_plot1,
                                 self.matplotlibwidget_compare1,
                                 self.comboBox_compare_plot1)
        self.compare_create_plot(self.widget_compare_plot2,
                                  self.matplotlibwidget_compare2,
                                  self.comboBox_compare_plot2)
        self.compare_create_plot(self.widget_compare_plot3,
                                  self.matplotlibwidget_compare3,
                                  self.comboBox_compare_plot3)
        self.compare_create_plot(self.widget_compare_plot4,
                                  self.matplotlibwidget_compare4,
                                  self.comboBox_compare_plot4)
        self.compare_create_plot(self.widget_compare_plot5,
                                  self.matplotlibwidget_compare5,
                                  self.comboBox_compare_plot5)
        self.compare_create_plot(self.widget_compare_plot6,
                                  self.matplotlibwidget_compare6,
                                  self.comboBox_compare_plot6)
        self.compare_create_plot(self.widget_compare_plot7,
                                 self.matplotlibwidget_compare7,
                                 self.comboBox_compare_plot7)
        self.compare_create_plot(self.widget_compare_plot8,
                                 self.matplotlibwidget_compare8,
                                 self.comboBox_compare_plot8)
        self.compare_create_plot(self.widget_compare_plot9,
                                 self.matplotlibwidget_compare9,
                                 self.comboBox_compare_plot9)

    def compare_pop_out_plot1(self):
        """
        Pops out plot1 in Comparison Cools in a PopOutPlotWindow

        :return: None
        """
        self.PopOutPlotWindow1 = PopOutPlotWindow()
        self.matplotlibwidget_pop_out_compare1 = MatplotlibWidget()

        self.compare_create_plot(self.PopOutPlotWindow1.widget_plot,
                                 self.matplotlibwidget_pop_out_compare1,
                                 self.comboBox_compare_plot1)

        self.PopOutPlotWindow1.show()

    def compare_pop_out_plot2(self):
        self.PopOutPlotWindow2 = PopOutPlotWindow()
        self.matplotlibwidget_pop_out_compare2 = MatplotlibWidget()

        self.compare_create_plot(self.PopOutPlotWindow2.widget_plot,
                                 self.matplotlibwidget_pop_out_compare2,
                                 self.comboBox_compare_plot2)

        self.PopOutPlotWindow2.show()

    def compare_pop_out_plot3(self):
        self.PopOutPlotWindow3 = PopOutPlotWindow()
        self.matplotlibwidget_pop_out_compare3 = MatplotlibWidget()

        self.compare_create_plot(self.PopOutPlotWindow3.widget_plot,
                                 self.matplotlibwidget_pop_out_compare3,
                                 self.comboBox_compare_plot3)

        self.PopOutPlotWindow3.show()

    def compare_pop_out_plot4(self):
        self.PopOutPlotWindow4 = PopOutPlotWindow()
        self.matplotlibwidget_pop_out_compare4 = MatplotlibWidget()

        self.compare_create_plot(self.PopOutPlotWindow4.widget_plot,
                                 self.matplotlibwidget_pop_out_compare4,
                                 self.comboBox_compare_plot4)

        self.PopOutPlotWindow4.show()

    def compare_pop_out_plot5(self):
        self.PopOutPlotWindow5 = PopOutPlotWindow()
        self.matplotlibwidget_pop_out_compare5 = MatplotlibWidget()

        self.compare_create_plot(self.PopOutPlotWindow5.widget_plot,
                                 self.matplotlibwidget_pop_out_compare5,
                                 self.comboBox_compare_plot5)

        self.PopOutPlotWindow5.show()

    def compare_pop_out_plot6(self):
        self.PopOutPlotWindow6 = PopOutPlotWindow()
        self.matplotlibwidget_pop_out_compare6 = MatplotlibWidget()

        self.compare_create_plot(self.PopOutPlotWindow6.widget_plot,
                                 self.matplotlibwidget_pop_out_compare6,
                                 self.comboBox_compare_plot6)

        self.PopOutPlotWindow6.show()

    def compare_pop_out_plot7(self):
        self.PopOutPlotWindow7 = PopOutPlotWindow()
        self.matplotlibwidget_pop_out_compare7 = MatplotlibWidget()

        self.compare_create_plot(self.PopOutPlotWindow7.widget_plot,
                                 self.matplotlibwidget_pop_out_compare7,
                                 self.comboBox_compare_plot7)

        self.PopOutPlotWindow7.show()

    def compare_pop_out_plot8(self):
        self.PopOutPlotWindow8 = PopOutPlotWindow()
        self.matplotlibwidget_pop_out_compare8 = MatplotlibWidget()

        self.compare_create_plot(self.PopOutPlotWindow8.widget_plot,
                                 self.matplotlibwidget_pop_out_compare8,
                                 self.comboBox_compare_plot8)

        self.PopOutPlotWindow8.show()

    def compare_pop_out_plot9(self):
        self.PopOutPlotWindow9 = PopOutPlotWindow()
        self.matplotlibwidget_pop_out_compare9 = MatplotlibWidget()

        self.compare_create_plot(self.PopOutPlotWindow9.widget_plot,
                                 self.matplotlibwidget_pop_out_compare9,
                                 self.comboBox_compare_plot9)

        self.PopOutPlotWindow9.show()





    def comboBox_compare_transistor_changed(self, comboBox_compare_transistor, comboBox_compare_v_g_on_transistor,
                                            comboBox_compare_v_g_off_transistor, lineEdit_compare_t_j_transistor,
                                            slider_compare_r_g_on_transistor, label_compare_r_g_on_value_transistor,
                                            slider_compare_r_g_off_transistor, label_compare_r_g_off_value_transistor):
        """
        Fills the comboBoxes for transistor with data based on the available data in the transistordatabase

        :return: None
        """
        transistor = tdb.load(comboBox_compare_transistor.currentText())
        comboBox_compare_v_g_on_transistor.clear()
        comboBox_compare_v_g_off_transistor.clear()

        available_v_g_on_transistor = [str(channel.v_g) for channel in transistor.switch.channel]
        available_v_g_on_transistor_cleared = []

        for v_g in available_v_g_on_transistor:
            if v_g not in available_v_g_on_transistor_cleared and v_g != "None":
                available_v_g_on_transistor_cleared.append(v_g)

        comboBox_compare_v_g_on_transistor.addItems(available_v_g_on_transistor_cleared)
        comboBox_compare_v_g_on_transistor.setCurrentText(
            str(max(channel.v_g for channel in transistor.switch.channel)))

        if transistor.type.lower() != "igbt":
            available_v_g_off_transistor = [str(channel.v_g) for channel in transistor.diode.channel]
            available_v_g_off_transistor_cleared = []

            for v_g in available_v_g_off_transistor:
                if v_g not in available_v_g_off_transistor_cleared and v_g != "None":
                    available_v_g_off_transistor_cleared.append(v_g)

            comboBox_compare_v_g_off_transistor.addItems(available_v_g_off_transistor_cleared)
            comboBox_compare_v_g_off_transistor.setCurrentText(
                str(min(channel.v_g for channel in transistor.diode.channel)))


        try:
            t_j_available_unfiltered = [i for i in [e_on.t_j for e_on in transistor.switch.e_on]]
            t_j_available_unfiltered = np.sort(t_j_available_unfiltered)
            t_j_available = []
            for i in t_j_available_unfiltered:
                if i not in t_j_available:
                    t_j_available.append(i)

            r_g_on_max_list = np.zeros_like(t_j_available)

            for i in range(len(t_j_available)):
                r_e_object_on = transistor.get_object_r_e_simplified(e_on_off_rr="e_on",
                                                                      t_j=t_j_available[i],
                                                                      v_g=max([i for i in [e_on.v_g for e_on in
                                                                                           transistor.switch.e_on] if
                                                                               i is not None]),
                                                                      v_supply=max(
                                                                          [i for i in [e_on.v_supply for e_on in
                                                                                       transistor.switch.e_on] if
                                                                           i is not None]),
                                                                     normalize_t_to_v=10)

                r_g_on_max_list[i] = np.amax(r_e_object_on.graph_r_e[0])*10000

            r_g_on_max = floor(10*min(r_g_on_max_list)/10000)/10
            slider_compare_r_g_on_transistor.setMinimum(0)
            slider_compare_r_g_on_transistor.setMaximum(int(r_g_on_max * 100))
            slider_compare_r_g_on_transistor.setValue(int(r_g_on_max * 100))
            label_compare_r_g_on_value_transistor.setText(str(r_g_on_max))

        except:
            try:
                r_g_on = max([i for i in [e_on.r_g for e_on in transistor.switch.e_on] if i is not None])
                slider_compare_r_g_on_transistor.setMinimum(int(r_g_on * 100))
                slider_compare_r_g_on_transistor.setMaximum(int(r_g_on * 100))
                #self.show_popup_message(f"No energy data for different turn on gate resistor for <b>{transistor.name}</b> available!")
            except:
                #self.show_popup_message(f"No turn-on energy data for <b>{transistor.name}</b> available!")
                slider_compare_r_g_on_transistor.setMinimum(0)
                slider_compare_r_g_on_transistor.setMaximum(0)

        try:
            t_j_available_unfiltered = [i for i in [e_off.t_j for e_off in transistor.switch.e_off]]
            t_j_available_unfiltered = np.sort(t_j_available_unfiltered)
            t_j_available = []
            for i in t_j_available_unfiltered:
                if i not in t_j_available:
                    t_j_available.append(i)

            r_g_off_max_list = np.zeros_like(t_j_available)

            for i in range(len(t_j_available)):
                r_e_object_off = transistor.get_object_r_e_simplified(e_on_off_rr="e_off",
                                                                       t_j=t_j_available[i],
                                                                       v_g=max([i for i in [e_off.v_g for e_off in
                                                                                            transistor.switch.e_off] if
                                                                                i is not None]),
                                                                       v_supply=max(
                                                                           [i for i in [e_off.v_supply for e_off in
                                                                                        transistor.switch.e_off] if
                                                                            i is not None]),
                                                                       normalize_t_to_v=10)
                r_g_off_max_list[i] = np.amax(r_e_object_off.graph_r_e[0])*10000


            r_g_off_max = floor(10*min(r_g_off_max_list)/10000)/10


            if transistor.type == "IGBT":
                t_j_available_unfiltered = [i for i in [e_rr.t_j for e_rr in transistor.diode.e_rr]]
                t_j_available_unfiltered = np.sort(t_j_available_unfiltered)
                t_j_available = []
                for i in t_j_available_unfiltered:
                    if i not in t_j_available:
                        t_j_available.append(i)

                r_g_rr_max_list = np.zeros_like(t_j_available)

                for i in range(len(t_j_available)):
                    r_e_object_rr = transistor.get_object_r_e_simplified(e_on_off_rr="e_rr",
                                                                          t_j=t_j_available[i],
                                                                          v_g=min([i for i in [e_rr.v_g for e_rr in
                                                                                               transistor.diode.e_rr]
                                                                                   if
                                                                                   i is not None]),
                                                                          v_supply=max([i for i in
                                                                                        [e_rr.v_supply for e_rr in
                                                                                         transistor.diode.e_rr] if
                                                                                        i is not None]),
                                                                          normalize_t_to_v=10)
                    r_g_rr_max_list[i] = np.amax(r_e_object_rr.graph_r_e[0])*10000

                r_g_rr_max = floor(10*min(r_g_rr_max_list)/10000)/10


            r_g_off_max = min(r_g_off_max, r_g_rr_max)

            slider_compare_r_g_off_transistor.setMinimum(0)
            slider_compare_r_g_off_transistor.setMaximum(int(r_g_off_max * 100))
            slider_compare_r_g_off_transistor.setValue(int(r_g_off_max * 100))
            label_compare_r_g_off_value_transistor.setText(str(r_g_off_max))
        except:
            try:
                r_g_off = max([i for i in [e_off.r_g for e_off in transistor.switch.e_off] if i is not None])
                slider_compare_r_g_off_transistor.setMinimum(int(r_g_off * 100))
                slider_compare_r_g_off_transistor.setMaximum(int(r_g_off * 100))
                #self.show_popup_message(f"No energy data for different turn off gate resistor for <b>{transistor.name}</b> available!")
            except:
                #self.show_popup_message(f"No turn-off energy data for <b>{transistor.name}</b> available!")
                slider_compare_r_g_off_transistor.setMinimum(0)
                slider_compare_r_g_off_transistor.setMaximum(0)



    def comboBox_compare_transistor1_changed(self):
        """
        Runs the function to fill the comboBoxes and configurate the sliders for the comparison tools tab for transistor1

        :return:
        """
        self.comboBox_compare_transistor_changed(self.comboBox_compare_transistor1,
                                                 self.comboBox_compare_v_g_on_transistor1,
                                                 self.comboBox_compare_v_g_off_transistor1,
                                                 self.lineEdit_compare_t_j_transistor1,
                                                 self.slider_compare_r_g_on_transistor1,
                                                 self.label_compare_r_g_on_value_transistor1,
                                                 self.slider_compare_r_g_off_transistor1,
                                                 self.label_compare_r_g_off_value_transistor1)

    def comboBox_compare_transistor2_changed(self):
        """
        Runs the function to fill the comboBoxes and configurate the sliders for the comparison tools tab for transistor2

        :return:
        """
        self.comboBox_compare_transistor_changed(self.comboBox_compare_transistor2,
                                                 self.comboBox_compare_v_g_on_transistor2,
                                                 self.comboBox_compare_v_g_off_transistor2,
                                                 self.lineEdit_compare_t_j_transistor2,
                                                 self.slider_compare_r_g_on_transistor2,
                                                 self.label_compare_r_g_on_value_transistor2,
                                                 self.slider_compare_r_g_off_transistor2,
                                                 self.label_compare_r_g_off_value_transistor2)

    def comboBox_compare_transistor3_changed(self):
        """
        Runs the function to fill the comboBoxes and configurate the sliders for the comparison tools tab for transistor3

        :return:
        """
        self.comboBox_compare_transistor_changed(self.comboBox_compare_transistor3,
                                                 self.comboBox_compare_v_g_on_transistor3,
                                                 self.comboBox_compare_v_g_off_transistor3,
                                                 self.lineEdit_compare_t_j_transistor3,
                                                 self.slider_compare_r_g_on_transistor3,
                                                 self.label_compare_r_g_on_value_transistor3,
                                                 self.slider_compare_r_g_off_transistor3,
                                                 self.label_compare_r_g_off_value_transistor3)



    def slider_compare_r_g_value_changed(self):
        """
        Sets the labels below the sliders to choose gate resistors to show currently selected values

        :return:
        """
        self.label_compare_r_g_on_value_transistor1.setText(str(round(self.slider_compare_r_g_on_transistor1.value()/100, 1)))
        self.label_compare_r_g_off_value_transistor1.setText(str(round(self.slider_compare_r_g_off_transistor1.value() / 100, 1)))
        self.label_compare_r_g_on_value_transistor2.setText(str(round(self.slider_compare_r_g_on_transistor2.value()/100, 1)))
        self.label_compare_r_g_off_value_transistor2.setText(str(round(self.slider_compare_r_g_off_transistor2.value() / 100, 1)))
        self.label_compare_r_g_on_value_transistor3.setText(str(round(self.slider_compare_r_g_on_transistor3.value() / 100, 1)))
        self.label_compare_r_g_off_value_transistor3.setText(str(round(self.slider_compare_r_g_off_transistor3.value() / 100, 1)))

    ###TOPOLOGY CALCULATOR###

    def comboBox_topology_topology_changed(self):
        """
        Sets the picture which shows the circuit diagram of selected topology and resizes it so that will be shown
        correctly

        :return: None
        """

        if self.comboBox_topology_topology.currentText() == "Buck-Converter":
            self.label_picture_topology.setMaximumSize(QtCore.QSize(450, 115))
            self.label_picture_topology.setPixmap(QtGui.QPixmap(resource_path("buck_converter_schematic.png")))
        elif self.comboBox_topology_topology.currentText() == "Boost-Converter":
            self.label_picture_topology.setMaximumSize(QtCore.QSize(450, 110))
            self.label_picture_topology.setPixmap(QtGui.QPixmap(resource_path("boost_converter_schematic.png")))
        elif self.comboBox_topology_topology.currentText() == "Buck-Boost-Converter":
            self.label_picture_topology.setMaximumSize(QtCore.QSize(450, 115))
            self.label_picture_topology.setPixmap(QtGui.QPixmap(resource_path("buck_boost_converter_schematic.png")))

    def comboBox_topology_plot_line_contour_changed(self, comboBox_topology_plot_y_axis, comboBox_topology_plot_z_axis, comboBox_topology_plot_line_contour):
        """
        Fills the ComboBoxes for y-axis and z-axis with variables depending on whether line or contour is selected

        :param comboBox_topology_plot_y_axis: ComboBox to choose variable for x-axis
        :param comboBox_topology_plot_z_axis: ComboBox to choose variable for y-axis
        :param comboBox_topology_plot_line_contour: ComboBox to choose between line-plot and contour-plot
        :return: None
        """
        current_y_axis = comboBox_topology_plot_y_axis.currentText()
        current_z_axis = comboBox_topology_plot_z_axis.currentText()
        self.comboBox_topology_plots_x_axis_changed()
        if comboBox_topology_plot_line_contour.currentText() == "Line":
            comboBox_topology_plot_y_axis.setCurrentText(current_z_axis)
        elif comboBox_topology_plot_line_contour.currentText() == "Contour":
            comboBox_topology_plot_z_axis.setCurrentText(current_y_axis)


    def comboBox_topology_plots_line_contour_changed(self):
        """
        Runs the function "comboBox_topology_plot_line_contour_changed" with the the ComboBoxes of all plots

        :return: None
        """
        self.comboBox_topology_plot_line_contour_changed(self.comboBox_topology_plot1_y_axis,
                                                         self.comboBox_topology_plot1_z_axis,
                                                         self.comboBox_topology_plot1_line_contour)
        self.comboBox_topology_plot_line_contour_changed(self.comboBox_topology_plot2_y_axis,
                                                         self.comboBox_topology_plot2_z_axis,
                                                         self.comboBox_topology_plot2_line_contour)
        self.comboBox_topology_plot_line_contour_changed(self.comboBox_topology_plot3_y_axis,
                                                         self.comboBox_topology_plot3_z_axis,
                                                         self.comboBox_topology_plot3_line_contour)
        self.comboBox_topology_plot_line_contour_changed(self.comboBox_topology_plot4_y_axis,
                                                         self.comboBox_topology_plot4_z_axis,
                                                         self.comboBox_topology_plot4_line_contour)
        self.comboBox_topology_plot_line_contour_changed(self.comboBox_topology_plot5_y_axis,
                                                         self.comboBox_topology_plot5_z_axis,
                                                         self.comboBox_topology_plot5_line_contour)
        self.comboBox_topology_plot_line_contour_changed(self.comboBox_topology_plot6_y_axis,
                                                         self.comboBox_topology_plot6_z_axis,
                                                         self.comboBox_topology_plot6_line_contour)


    def comboBox_topology_plot_x_axis_changed(self, comboBox_topology_plot_x_axis, comboBox_topology_plot_y_axis,
                                              comboBox_topology_plot_z_axis, comboBox_topology_plot_line_contour):
        """
        Fills the ComboBoxes to choose the variables for y-axis and z-axis based on the selection of the variable for
        the x-axis and whether line or contour is selected

        :param comboBox_topology_plot_x_axis: ComboBox to choose variable for x-axis
        :param comboBox_topology_plot_y_axis: ComboBox to choose variable for y-axis
        :param comboBox_topology_plot_z_axis: ComboBox to choose variable for z-axis
        :param comboBox_topology_plot_line_contour: ComboBox to choose between line-plot and contour-plot
        :return: None
        """

        items_comboBox_y_z_axis = ["RMS Current Transistor1 [A]",
                                 "RMS Current Diode Transistor2 [A]",
                                 "Mean Current Transistor1 [A]",
                                 "Mean Current Diode Transistor2 [A]",
                                 "RMS Inductor Current [A]",
                                 "Mean Inductor Current [A]",
                                 "Peak Current [A]",
                                 "Conduction Losses Transistor1 [W]",
                                 "Conduction Losses Diode Transistor2 [W]",
                                 "Total Conduction Losses [W]",
                                 "Turn-on Switching Losses Transistor1 [W]",
                                 "Turn-off Switching Losses Transistor1 [W]",
                                 "Reverse Recovery Losses Diode Transistor2 [W]",
                                 "Total Switching Losses Transistor1 [W]",
                                 "Total Power Losses Transistor1 [W]",
                                 "Total Switching Losses [W]",
                                 "Temperature Switch Transistor1 [°C]",
                                 "Temperature Diode Transistor2 [°C]"]

        current_y_axis = comboBox_topology_plot_y_axis.currentText()
        comboBox_topology_plot_y_axis.clear()
        current_z_axis = comboBox_topology_plot_z_axis.currentText()
        comboBox_topology_plot_z_axis.clear()
        if comboBox_topology_plot_line_contour.currentText() == "Line":
            comboBox_topology_plot_y_axis.addItems(items_comboBox_y_z_axis)
            comboBox_topology_plot_y_axis.setCurrentText(current_y_axis)
        elif comboBox_topology_plot_line_contour.currentText() == "Contour":
            items_comboBox_y_axis = ["Vin [V]", "Vout [V]", "Output Power [W]", "Frequency [kHz]", "Zeta = f*L"]
            items_comboBox_y_axis.remove(comboBox_topology_plot_x_axis.currentText())
            comboBox_topology_plot_y_axis.addItems(items_comboBox_y_axis)
            comboBox_topology_plot_y_axis.setCurrentText(current_y_axis)
            comboBox_topology_plot_z_axis.addItems(items_comboBox_y_z_axis)
            comboBox_topology_plot_z_axis.setCurrentText(current_z_axis)


    def comboBox_topology_plots_x_axis_changed(self):
        """
        Runs the function "comboBox_topology_plot_x_axis_changed" with the ComboBoxes of all plots

        :return: None
        """
        self.comboBox_topology_plot_x_axis_changed(self.comboBox_topology_plot1_x_axis,
                                                   self.comboBox_topology_plot1_y_axis,
                                                   self.comboBox_topology_plot1_z_axis,
                                                   self.comboBox_topology_plot1_line_contour)
        self.comboBox_topology_plot_x_axis_changed(self.comboBox_topology_plot2_x_axis,
                                                   self.comboBox_topology_plot2_y_axis,
                                                   self.comboBox_topology_plot2_z_axis,
                                                   self.comboBox_topology_plot2_line_contour)
        self.comboBox_topology_plot_x_axis_changed(self.comboBox_topology_plot3_x_axis,
                                                   self.comboBox_topology_plot3_y_axis,
                                                   self.comboBox_topology_plot3_z_axis,
                                                   self.comboBox_topology_plot3_line_contour)
        self.comboBox_topology_plot_x_axis_changed(self.comboBox_topology_plot4_x_axis,
                                                   self.comboBox_topology_plot4_y_axis,
                                                   self.comboBox_topology_plot4_z_axis,
                                                   self.comboBox_topology_plot4_line_contour)
        self.comboBox_topology_plot_x_axis_changed(self.comboBox_topology_plot5_x_axis,
                                                   self.comboBox_topology_plot5_y_axis,
                                                   self.comboBox_topology_plot5_z_axis,
                                                   self.comboBox_topology_plot5_line_contour)
        self.comboBox_topology_plot_x_axis_changed(self.comboBox_topology_plot6_x_axis,
                                                   self.comboBox_topology_plot6_y_axis,
                                                   self.comboBox_topology_plot6_z_axis,
                                                   self.comboBox_topology_plot6_line_contour)



    def comboBox_topology_transistor_changed(self, comboBox_topology_transistor, comboBox_topology_v_g_on_transistor,
                                             slider_topology_r_g_on_transistor,
                                             label_topology_slider_r_g_on_value_transistor,
                                             slider_topology_r_g_off_transistor,
                                             label_topology_slider_r_g_off_value_transistor):
        """
        Fills the ComboBox to choose the gate voltage for the transistor and sets minimum and maximum value for the sliders
        to choose the gate resistors for transistor based on the available data stored in the transistordatabase

        :return: None
        """
        transistor = tdb.load(comboBox_topology_transistor.currentText())

        comboBox_topology_v_g_on_transistor.clear()
        available_v_g_on_transistor = [str(channel.v_g) for channel in transistor.switch.channel]
        available_v_g_on_transistor_cleared = [available_v_g_on_transistor[i] for i in
                                               range(len(available_v_g_on_transistor)) if
                                               i == available_v_g_on_transistor.index(
                                                   available_v_g_on_transistor[i])]
        comboBox_topology_v_g_on_transistor.addItems(available_v_g_on_transistor_cleared)
        comboBox_topology_v_g_on_transistor.setCurrentText(
            str(max(channel.v_g for channel in transistor.switch.channel)))

        try:
            r_e_object_on = transistor.get_object_r_e_simplified(e_on_off_rr="e_on",
                                                                  t_j=max([i for i in [e_on.t_j for e_on in
                                                                                       transistor.switch.e_on] if
                                                                           i is not None]),
                                                                  v_g=max([i for i in [e_on.v_g for e_on in
                                                                                       transistor.switch.e_on] if
                                                                           i is not None]),
                                                                  v_supply=max([i for i in [e_on.v_supply for e_on in
                                                                                            transistor.switch.e_on] if
                                                                                i is not None]),
                                                                  normalize_t_to_v=10)
            r_g_on_max = floor(10*(np.amax(r_e_object_on.graph_r_e[0])))/10
            slider_topology_r_g_on_transistor.setMinimum(0)
            slider_topology_r_g_on_transistor.setMaximum(int(r_g_on_max * 100))
            slider_topology_r_g_on_transistor.setValue(int(r_g_on_max * 100))
            label_topology_slider_r_g_on_value_transistor.setText(str(round(r_g_on_max, 1)))

        except:
            try:
                r_g_on_max = max([i for i in [e_on.r_g for e_on in transistor.switch.e_on] if i is not None])
                slider_topology_r_g_on_transistor.setMinimum(int(r_g_on_max * 100))
                slider_topology_r_g_on_transistor.setMaximum(int(r_g_on_max * 100))
                if not self.start:
                    self.show_popup_message(f"No energy data for different turn on gate resistor for <b>{transistor.name}</b> available!")
            except:
                if not self.start:
                    self.show_popup_message(f"No turn-on energy data for <b>{transistor.name}</b> available!")
                slider_topology_r_g_on_transistor.setMinimum(0)
                slider_topology_r_g_on_transistor.setMaximum(0)

        try:
            r_e_object_off = transistor.get_object_r_e_simplified(e_on_off_rr="e_off",
                                                                   t_j=max([i for i in [e_off.t_j for e_off in
                                                                                        transistor.switch.e_off] if
                                                                            i is not None]),
                                                                   v_g=min([i for i in [e_off.v_g for e_off in
                                                                                        transistor.switch.e_off] if
                                                                            i is not None]),
                                                                   v_supply=max([i for i in
                                                                                 [e_off.v_supply for e_off in
                                                                                  transistor.switch.e_off] if
                                                                                 i is not None]),
                                                                   normalize_t_to_v=10)
            r_g_off_max = floor(10*np.amax(r_e_object_off.graph_r_e[0]))/10
            slider_topology_r_g_off_transistor.setMinimum(0)

            if transistor.type == "IGBT":
                r_e_object_rr = transistor.get_object_r_e_simplified(e_on_off_rr="e_rr",
                                                                      t_j=max([i for i in [e_rr.t_j for e_rr in
                                                                                           transistor.diode.e_rr] if
                                                                               i is not None]),
                                                                      v_g=min([i for i in [e_rr.v_g for e_rr in
                                                                                           transistor.diode.e_rr] if
                                                                               i is not None]),
                                                                      v_supply=max([i for i in
                                                                                    [e_rr.v_supply for e_rr in
                                                                                     transistor.diode.e_rr] if
                                                                                    i is not None]),
                                                                      normalize_t_to_v=10)
                r_g_rr_max = floor(10*(np.amax(r_e_object_rr.graph_r_e[0])))/10
                r_g_max_off_rr1 = min(r_g_off_max, r_g_rr_max)
                slider_topology_r_g_off_transistor.setMaximum(int(r_g_max_off_rr1 * 100))
                slider_topology_r_g_off_transistor.setValue(int(r_g_max_off_rr1 * 100))
                label_topology_slider_r_g_off_value_transistor.setText(str(round(r_g_max_off_rr1, 1)))
            else:
                slider_topology_r_g_off_transistor.setMaximum(int(r_g_off_max * 100))
                slider_topology_r_g_off_transistor.setValue(int(r_g_off_max * 100))
                label_topology_slider_r_g_off_value_transistor.setText(str(round(r_g_off_max, 1)))
        except:
            try:
                r_g_off_max = max([i for i in [e_off.r_g for e_off in transistor.switch.e_off] if i is not None])
                slider_topology_r_g_off_transistor.setMinimum(int(r_g_off_max * 100))
                slider_topology_r_g_off_transistor.setMaximum(int(r_g_off_max * 100))
                if not self.start:
                    self.show_popup_message(f"No energy data for different turn off gate resistor for <b>{transistor.name}</b> available!")
            except:
                slider_topology_r_g_off_transistor.setMinimum(0)
                slider_topology_r_g_off_transistor.setMaximum(0)
                if not self.start:
                    self.show_popup_message(f"No turn-off energy data for <b>{transistor.name}</b> available!")
        self.start=None    


    def comboBox_topology_transistor1_changed(self):
        """
        Runs the function to fill the comboBoxes and configurate the sliders for the topology calculator tab for transistor1

        :return: None
        """
        print("we are before the comboBox_topology_transistor_changed ")
        self.comboBox_topology_transistor_changed(self.comboBox_topology_transistor1,
                                                   self.comboBox_topology_v_g_on_transistor1,
                                                   self.slider_topology_r_g_on_transistor1,
                                                   self.label_topology_slider_r_g_on_value_transistor1,
                                                   self.slider_topology_r_g_off_transistor1,
                                                   self.label_topology_slider_r_g_off_value_transistor1)



    def slider_topology_r_g_value_changed(self):
        """
        Sets the labels below the sliders to choose gate resistors to show currently selected values

        :return:
        """
        self.label_topology_slider_r_g_on_value_transistor1.setText(str(round(self.slider_topology_r_g_on_transistor1.value()/100, 1)))
        self.label_topology_slider_r_g_off_value_transistor1.setText(str(round(self.slider_topology_r_g_off_transistor1.value()/100, 1)))


    def new_annotation(self, axis):
        """
        Creates an annotation and adds it to a matplotlibwidget

        :param axis: matplotlibwidget axis to put the annotation on
        :return: annotation object
        """
        annotation = axis.annotate("", xy=(0, 0), xytext=(-110, 30), textcoords="offset pixels",
                                   bbox=dict(boxstyle="square", fc="linen", ec="k", lw="1"),
                                   arrowprops=dict(arrowstyle="-|>"))
        return annotation

    def click_event(self, button, xdata, ydata, matplotlibwidget, annotations_list):
        """
        Creates an annotation for an embedded matplotlibwidget graph on left click and removes last created annotation on
        right click

        :param button: clicked button
        :param xdata: x-data of matplotlib graph
        :param ydata: y-data of matplotlib graph
        :param matplotlibwidget: matplotlibwidget object
        :param annotations_list: list to store added annotations
        :return: None
        """
        if str(button) == "MouseButton.LEFT":
            click_annotation = self.new_annotation(matplotlibwidget.axis)
            annotations_list.append(click_annotation)
            click_annotation.xy = (xdata, ydata)
            text = f"({round(xdata, 2)}, {round(ydata, 2)})"
            click_annotation.set_text(text)
            click_annotation.set_visible(True)
        elif str(button) == "MouseButton.RIGHT":
            annotations_list[-1].remove()
            annotations_list.pop()
        matplotlibwidget.figure.canvas.draw_idle()


    def topology_create_plot(self, widget_topology_plot, matplotlibwidget, comboBox_topology_plot_x_axis, comboBox_topology_plot_y_axis, comboBox_topology_plot_z_axis, comboBox_topology_line_contour, converter):
        """
        Adds a Matplotlib figure to a QWidget and creates a plot based on all the possible inputs and selections using
        the calculation-functions from buck_converter_functions.py, boost_converter_functions.py and buck_boost_converter_functions.py.

        :param widget_topology_plot: widget for the Matplotlib figure
        :param matplotlibwidget: Matplotlib figure
        :param comboBox_topology_plot_x_axis: ComboBox for selection of the variable for the x-axis
        :param comboBox_topology_plot_y_axis: ComboBox for selection of the variable for the y-axis
        :param comboBox_topology_plot_z_axis: ComboBox for selection of the variable for the z-axis
        :param comboBox_topology_line_contour: ComboBox for selection between line-plot and contour-plot
        :param converter: "buck_converter", "boost_converter" or "buck_boost_converter"
        :return: None
        """

        annotations_list = []
        def clicked(event):
            if event.dblclick:
                self.click_event(event.button, event.xdata, event.ydata, matplotlibwidget, annotations_list)



        self.layout = QVBoxLayout(widget_topology_plot)
        self.layout.addWidget(matplotlibwidget)
        matplotlibwidget.axis.clear()
        try:
            matplotlibwidget.axis_cm.remove()
        except:
            pass

        transistor1 = tdb.load(self.comboBox_topology_transistor1.currentText())
        transistor2 = tdb.load(self.comboBox_topology_transistor2.currentText())
        v_g_on1 = float(self.comboBox_topology_v_g_on_transistor1.currentText())
        r_g_on1 = float(self.label_topology_slider_r_g_on_value_transistor1.text())
        r_g_off1 = float(self.label_topology_slider_r_g_off_value_transistor1.text())

        try:
            t_heatsink = float(self.lineEdit_topology_temperature_heatsink.text())
            r_th_heatsink = float(self.lineEdit_topology_thermal_resistance_heatsink.text())
        except:
            pass

        try:
            transistor1 = transistor1.parallel_transistors(int(self.lineEdit_topology_number_parallel_transistor1.text()))
            transistor2 = transistor2.parallel_transistors(int(self.lineEdit_topology_number_parallel_transistor2.text()))

            if comboBox_topology_line_contour.currentText() == "Contour":
                if comboBox_topology_plot_x_axis.currentText() == "Zeta = f*L":
                    vec_x_axis = np.linspace(float(self.lineEdit_topology_zeta_min.text()),
                                             float(self.lineEdit_topology_zeta_max.text()), 100)
                if comboBox_topology_plot_x_axis.currentText() == "Vin [V]":
                    vec_x_axis = np.linspace(float(self.lineEdit_topology_v_in_min.text()),
                                             float(self.lineEdit_topology_v_in_max.text()), 100)
                if comboBox_topology_plot_x_axis.currentText() == "Vout [V]":
                    vec_x_axis = np.linspace(float(self.lineEdit_topology_v_out_min.text()),
                                             float(self.lineEdit_topology_v_out_max.text()), 100)
                if comboBox_topology_plot_x_axis.currentText() == "Output Power [W]":
                    vec_x_axis = np.linspace(float(self.lineEdit_topology_output_power_min.text()),
                                             float(self.lineEdit_topology_output_power_max.text()), 100)
                if comboBox_topology_plot_x_axis.currentText() == "Frequency [kHz]":
                    vec_x_axis = np.linspace(float(self.lineEdit_topology_frequency_min.text()),
                                             float(self.lineEdit_topology_frequency_max.text()), 100)

                if comboBox_topology_plot_y_axis.currentText() == "Zeta = f*L":
                    vec_y_axis = np.linspace(float(self.lineEdit_topology_zeta_min.text()),
                                             float(self.lineEdit_topology_zeta_max.text()), 100)
                if comboBox_topology_plot_y_axis.currentText() == "Vin [V]":
                    vec_y_axis = np.linspace(float(self.lineEdit_topology_v_in_min.text()),
                                             float(self.lineEdit_topology_v_in_max.text()), 100)
                if comboBox_topology_plot_y_axis.currentText() == "Vout [V]":
                    vec_y_axis = np.linspace(float(self.lineEdit_topology_v_out_min.text()),
                                             float(self.lineEdit_topology_v_out_max.text()), 100)
                if comboBox_topology_plot_y_axis.currentText() == "Output Power [W]":
                    vec_y_axis = np.linspace(float(self.lineEdit_topology_output_power_min.text()),
                                             float(self.lineEdit_topology_output_power_max.text()), 100)
                if comboBox_topology_plot_y_axis.currentText() == "Frequency [kHz]":
                    vec_y_axis = np.linspace(float(self.lineEdit_topology_frequency_min.text()),
                                             float(self.lineEdit_topology_frequency_max.text()), 100)

                m_x, m_y = np.meshgrid(vec_x_axis, vec_y_axis)

                if self.lineEdit_topology_zeta.text() != "":
                    m_zeta = np.full_like(m_x, float(self.lineEdit_topology_zeta.text()))
                if self.lineEdit_topology_v_in.text() != "":
                    m_v_in = np.full_like(m_x, float(self.lineEdit_topology_v_in.text()))
                if self.lineEdit_topology_v_out.text() != "":
                    m_v_out = np.full_like(m_x, float(self.lineEdit_topology_v_out.text()))
                if self.lineEdit_topology_output_power.text() != "":
                    m_output_power = np.full_like(m_x, float(self.lineEdit_topology_output_power.text()))
                if self.lineEdit_topology_frequency.text() != "":
                    m_frequency = np.full_like(m_x, float(self.lineEdit_topology_frequency.text()))

                if comboBox_topology_plot_x_axis.currentText() == "Zeta = f*L":
                    m_zeta = m_x
                if comboBox_topology_plot_x_axis.currentText() == "Vin [V]":
                    m_v_in = m_x
                if comboBox_topology_plot_x_axis.currentText() == "Vout [V]":
                    m_v_out = m_x
                if comboBox_topology_plot_x_axis.currentText() == "Output Power [W]":
                    m_output_power = m_x
                if comboBox_topology_plot_x_axis.currentText() == "Frequency [kHz]":
                    m_frequency = m_x

                if comboBox_topology_plot_y_axis.currentText() == "Zeta = f*L":
                    m_zeta = m_y
                if comboBox_topology_plot_y_axis.currentText() == "Vin [V]":
                    m_v_in = m_y
                if comboBox_topology_plot_y_axis.currentText() == "Vout [V]":
                    m_v_out = m_y
                if comboBox_topology_plot_y_axis.currentText() == "Output Power [W]":
                    m_output_power = m_y
                if comboBox_topology_plot_y_axis.currentText() == "Frequency [kHz]":
                    m_frequency = m_y

                if comboBox_topology_plot_z_axis.currentText() == "RMS Current Transistor1 [A]":
                    m_z = converter.f_m_i1_rms(zeta=m_zeta,
                                               v_in=m_v_in,
                                               v_out=m_v_out,
                                               p_out=m_output_power,
                                               v_g_on1=v_g_on1,
                                               transistor1=transistor1,
                                               transistor2=transistor2)
                if comboBox_topology_plot_z_axis.currentText() == "RMS Current Diode Transistor2 [A]":
                    m_z = converter.f_m_i2_rms(zeta=m_zeta,
                                               v_in=m_v_in,
                                               v_out=m_v_out,
                                               p_out=m_output_power,
                                               v_g_on1=v_g_on1,
                                               transistor1=transistor1,
                                               transistor2=transistor2)
                if comboBox_topology_plot_z_axis.currentText() == "Mean Current Transistor1 [A]":
                    m_z = converter.f_m_i1_mean(zeta=m_zeta,
                                                v_in=m_v_in,
                                                v_out=m_v_out,
                                                p_out=m_output_power,
                                                v_g_on1=v_g_on1,
                                                transistor1=transistor1,
                                                transistor2=transistor2)
                if comboBox_topology_plot_z_axis.currentText() == "Mean Current Diode Transistor2 [A]":
                    m_z = converter.f_m_i2_mean(zeta=m_zeta,
                                                v_in=m_v_in,
                                                v_out=m_v_out,
                                                p_out=m_output_power,
                                                v_g_on1=v_g_on1,
                                                transistor1=transistor1,
                                                transistor2=transistor2)
                if comboBox_topology_plot_z_axis.currentText() == "RMS Inductor Current [A]":
                    m_z = converter.f_m_i_l_rms(zeta=m_zeta,
                                                v_in=m_v_in,
                                                v_out=m_v_out,
                                                p_out=m_output_power,
                                                v_g_on1=v_g_on1,
                                                transistor1=transistor1,
                                                transistor2=transistor2)
                if comboBox_topology_plot_z_axis.currentText() == "Mean Inductor Current [A]":
                    m_z = converter.f_m_i_l_mean(zeta=m_zeta,
                                                 v_in=m_v_in,
                                                 v_out=m_v_out,
                                                 p_out=m_output_power,
                                                 v_g_on1=v_g_on1,
                                                 transistor1=transistor1,
                                                 transistor2=transistor2)
                if comboBox_topology_plot_z_axis.currentText() == "Peak Current [A]":
                    m_z = converter.f_m_i_peak(zeta=m_zeta,
                                               v_in=m_v_in,
                                               v_out=m_v_out,
                                               p_out=m_output_power,
                                               v_g_on1=v_g_on1,
                                               transistor1=transistor1,
                                               transistor2=transistor2)
                if comboBox_topology_plot_z_axis.currentText() == "Conduction Losses Transistor1 [W]":
                    m_z = converter.f_m_conduction_losses1(zeta=m_zeta,
                                                           v_in=m_v_in,
                                                           v_out=m_v_out,
                                                           p_out=m_output_power,
                                                           v_g_on1=v_g_on1,
                                                           transistor1=transistor1,
                                                           transistor2=transistor2)
                if comboBox_topology_plot_z_axis.currentText() == "Conduction Losses Diode Transistor2 [W]":
                    m_z = converter.f_m_conduction_losses2(zeta=m_zeta,
                                                           v_in=m_v_in,
                                                           v_out=m_v_out,
                                                           p_out=m_output_power,
                                                           v_g_on1=v_g_on1,
                                                           transistor1=transistor1,
                                                           transistor2=transistor2)
                if comboBox_topology_plot_z_axis.currentText() == "Total Conduction Losses [W]":
                    m_z = converter.f_m_conduction_losses(zeta=m_zeta,
                                                          v_in=m_v_in,
                                                          v_out=m_v_out,
                                                          p_out=m_output_power,
                                                          v_g_on1=v_g_on1,
                                                          transistor1=transistor1,
                                                          transistor2=transistor2)
                if comboBox_topology_plot_z_axis.currentText() == "Turn-on Switching Losses Transistor1 [W]":
                    m_z = converter.f_m_p_on1(zeta=m_zeta,
                                              v_in=m_v_in,
                                              v_out=m_v_out,
                                              p_out=m_output_power,
                                              v_g_on1=v_g_on1,
                                              transistor1=transistor1,
                                              transistor2=transistor2,
                                              r_g_on1=r_g_on1,
                                              frequency=m_frequency)
                if comboBox_topology_plot_z_axis.currentText() == "Turn-off Switching Losses Transistor1 [W]":
                    m_z = converter.f_m_p_off1(zeta=m_zeta,
                                               v_in=m_v_in,
                                               v_out=m_v_out,
                                               p_out=m_output_power,
                                               v_g_on1=v_g_on1,
                                               transistor1=transistor1,
                                               transistor2=transistor2,
                                               r_g_off1=r_g_off1,
                                               frequency=m_frequency)
                if comboBox_topology_plot_z_axis.currentText() == "Reverse Recovery Losses Diode Transistor2 [W]":
                    m_z = converter.f_m_p_rr2(zeta=m_zeta,
                                              v_in=m_v_in,
                                              v_out=m_v_out,
                                              p_out=m_output_power,
                                              v_g_on1=v_g_on1,
                                              transistor1=transistor1,
                                              transistor2=transistor2,
                                              frequency=m_frequency)
                if comboBox_topology_plot_z_axis.currentText() == "Total Switching Losses Transistor1 [W]":
                    m_z = converter.f_m_p_on_off1(zeta=m_zeta,
                                                     v_in=m_v_in,
                                                     v_out=m_v_out,
                                                     p_out=m_output_power,
                                                     v_g_on1=v_g_on1,
                                                     transistor1=transistor1,
                                                     transistor2=transistor2,
                                                     r_g_on1=r_g_on1,
                                                     r_g_off1=r_g_off1,
                                                     frequency=m_frequency)
                if comboBox_topology_plot_z_axis.currentText() == "Total Switching Losses [W]":
                    m_z = converter.f_m_p_on_off_rr_1_2(zeta=m_zeta,
                                                        v_in=m_v_in,
                                                        v_out=m_v_out,
                                                        p_out=m_output_power,
                                                        v_g_on1=v_g_on1,
                                                        transistor1=transistor1,
                                                        transistor2=transistor2,
                                                        r_g_on1=r_g_on1,
                                                        r_g_off1=r_g_off1,
                                                        frequency=m_frequency)
                if comboBox_topology_plot_z_axis.currentText() == "Total Power Losses Transistor1 [W]":
                    m_z = converter.f_m_p1(zeta=m_zeta,
                                           v_in=m_v_in,
                                           v_out=m_v_out,
                                           p_out=m_output_power,
                                           v_g_on1=v_g_on1,
                                           transistor1=transistor1,
                                           transistor2=transistor2,
                                           r_g_on1=r_g_on1,
                                           r_g_off1=r_g_off1,
                                           frequency=m_frequency)
                if comboBox_topology_plot_z_axis.currentText() == "Temperature Switch Transistor1 [°C]":
                    m_z = converter.f_m_t_switch1(zeta=m_zeta,
                                                  v_in=m_v_in,
                                                  v_out=m_v_out,
                                                  p_out=m_output_power,
                                                  v_g_on1=v_g_on1,
                                                  r_g_on1=r_g_on1,
                                                  r_g_off1=r_g_off1,
                                                  t_heatsink=t_heatsink,
                                                  r_th_heatsink=r_th_heatsink,
                                                  frequency=m_frequency,
                                                  transistor1=transistor1,
                                                  transistor2=transistor2)
                if comboBox_topology_plot_z_axis.currentText() == "Temperature Diode Transistor2 [°C]":
                    m_z = converter.f_m_t_diode2(zeta=m_zeta,
                                                 v_in=m_v_in,
                                                 v_out=m_v_out,
                                                 p_out=m_output_power,
                                                 v_g_on1=v_g_on1,
                                                 t_heatsink=t_heatsink,
                                                 r_th_heatsink=r_th_heatsink,
                                                 frequency=m_frequency,
                                                 transistor1=transistor1,
                                                 transistor2=transistor2)

                plot = matplotlibwidget.axis.contourf(m_x, m_y, m_z, 100, cmap=cm.inferno)
                matplotlibwidget.divider = make_axes_locatable(matplotlibwidget.axis)
                matplotlibwidget.axis_cm = matplotlibwidget.divider.append_axes("right", size="3%", pad=0.03)
                matplotlibwidget.figure.colorbar(plot, cax=matplotlibwidget.axis_cm, format='%.2f')
                matplotlibwidget.axis.set_position([0.175, 0.15, 0.7, 0.75])
                matplotlibwidget.axis.ticklabel_format(useOffset=False)
                matplotlibwidget.axis.xaxis.set_major_formatter(FormatStrFormatter('%.2f'))
                matplotlibwidget.axis.yaxis.set_major_formatter(FormatStrFormatter('%.2f'))
                matplotlibwidget.axis.set(xlabel=comboBox_topology_plot_x_axis.currentText(),
                                          ylabel=comboBox_topology_plot_y_axis.currentText(),
                                          title=comboBox_topology_plot_z_axis.currentText())
                matplotlibwidget.figure.canvas.draw_idle()

                matplotlibwidget.cursor = Cursor(matplotlibwidget.axis, horizOn=True, vertOn=True, useblit=True,
                                                 color="Green",
                                                 linewidth=1)
                matplotlibwidget.figure.canvas.mpl_connect("button_press_event", clicked)

        except:
            matplotlibwidget.axis.clear()
            matplotlibwidget.figure.canvas.draw_idle()
            self.show_popup_message("Error: "+comboBox_topology_plot_z_axis.currentText()+" could not be plotted due to missing inputs or data!")


        try:
            if comboBox_topology_line_contour.currentText() == "Line":
                if comboBox_topology_plot_x_axis.currentText() == "Zeta = f*L":
                    vec_x_axis = np.linspace(float(self.lineEdit_topology_zeta_min.text()),
                                             float(self.lineEdit_topology_zeta_max.text()), 100)
                if comboBox_topology_plot_x_axis.currentText() == "Vin [V]":
                    vec_x_axis = np.linspace(float(self.lineEdit_topology_v_in_min.text()),
                                             float(self.lineEdit_topology_v_in_max.text()), 100)
                if comboBox_topology_plot_x_axis.currentText() == "Vout [V]":
                    vec_x_axis = np.linspace(float(self.lineEdit_topology_v_out_min.text()),
                                             float(self.lineEdit_topology_v_out_max.text()), 100)
                if comboBox_topology_plot_x_axis.currentText() == "Output Power [W]":
                    vec_x_axis = np.linspace(float(self.lineEdit_topology_output_power_min.text()),
                                             float(self.lineEdit_topology_output_power_max.text()), 100)
                if comboBox_topology_plot_x_axis.currentText() == "Frequency [kHz]":
                    vec_x_axis = np.linspace(float(self.lineEdit_topology_frequency_min.text()),
                                             float(self.lineEdit_topology_frequency_max.text()), 100)

                if self.lineEdit_topology_zeta.text() != "":
                    vec_zeta = np.full_like(vec_x_axis, float(self.lineEdit_topology_zeta.text()))
                if self.lineEdit_topology_v_in.text() != "":
                    vec_v_in = np.full_like(vec_x_axis, float(self.lineEdit_topology_v_in.text()))
                if self.lineEdit_topology_v_out.text() != "":
                    vec_v_out = np.full_like(vec_x_axis, float(self.lineEdit_topology_v_out.text()))
                if self.lineEdit_topology_output_power.text() != "":
                    vec_output_power = np.full_like(vec_x_axis, float(self.lineEdit_topology_output_power.text()))
                if self.lineEdit_topology_frequency.text() != "":
                    vec_frequency = np.full_like(vec_x_axis, float(self.lineEdit_topology_frequency.text()))

                if comboBox_topology_plot_x_axis.currentText() == "Zeta = f*L":
                    vec_zeta = vec_x_axis
                if comboBox_topology_plot_x_axis.currentText() == "Vin [V]":
                    vec_v_in = vec_x_axis
                if comboBox_topology_plot_x_axis.currentText() == "Vout [V]":
                    vec_v_out = vec_x_axis
                if comboBox_topology_plot_x_axis.currentText() == "Output Power [W]":
                    vec_output_power = vec_x_axis
                if comboBox_topology_plot_x_axis.currentText() == "Frequency [kHz]":
                    vec_frequency = vec_x_axis

                if comboBox_topology_plot_y_axis.currentText() == "RMS Current Transistor1 [A]":
                    vec_y_axis = converter.f_vec_i1_rms(zeta=vec_zeta,
                                                        v_in=vec_v_in,
                                                        v_out=vec_v_out,
                                                        p_out=vec_output_power,
                                                        v_g_on1=v_g_on1,
                                                        transistor1=transistor1,
                                                        transistor2=transistor2)
                if comboBox_topology_plot_y_axis.currentText() == "RMS Current Diode Transistor2 [A]":
                    vec_y_axis = converter.f_vec_i2_rms(zeta=vec_zeta,
                                                        v_in=vec_v_in,
                                                        v_out=vec_v_out,
                                                        p_out=vec_output_power,
                                                        v_g_on1=v_g_on1,
                                                        transistor1=transistor1,
                                                        transistor2=transistor2)
                if comboBox_topology_plot_y_axis.currentText() == "Mean Current Transistor1 [A]":
                    vec_y_axis = converter.f_vec_i1_mean(zeta=vec_zeta,
                                                         v_in=vec_v_in,
                                                         v_out=vec_v_out,
                                                         p_out=vec_output_power,
                                                         v_g_on1=v_g_on1,
                                                         transistor1=transistor1,
                                                         transistor2=transistor2)
                if comboBox_topology_plot_y_axis.currentText() == "Mean Current Diode Transistor2 [A]":
                    vec_y_axis = converter.f_vec_i2_mean(zeta=vec_zeta,
                                                         v_in=vec_v_in,
                                                         v_out=vec_v_out,
                                                         p_out=vec_output_power,
                                                         v_g_on1=v_g_on1,
                                                         transistor1=transistor1,
                                                         transistor2=transistor2)
                if comboBox_topology_plot_y_axis.currentText() == "RMS Inductor Current [A]":
                    vec_y_axis = converter.f_vec_i_l_rms(zeta=vec_zeta,
                                                         v_in=vec_v_in,
                                                         v_out=vec_v_out,
                                                         p_out=vec_output_power,
                                                         v_g_on1=v_g_on1,
                                                         transistor1=transistor1,
                                                         transistor2=transistor2)
                if comboBox_topology_plot_y_axis.currentText() == "Mean Inductor Current [A]":
                    vec_y_axis = converter.f_vec_i_l_mean(zeta=vec_zeta,
                                                          v_in=vec_v_in,
                                                          v_out=vec_v_out,
                                                          p_out=vec_output_power,
                                                          v_g_on1=v_g_on1,
                                                          transistor1=transistor1,
                                                          transistor2=transistor2)
                if comboBox_topology_plot_y_axis.currentText() == "Peak Current [A]":
                    vec_y_axis = converter.f_vec_i_peak(zeta=vec_zeta,
                                                        v_in=vec_v_in,
                                                        v_out=vec_v_out,
                                                        p_out=vec_output_power,
                                                        v_g_on1=v_g_on1,
                                                        transistor1=transistor1,
                                                        transistor2=transistor2)
                if comboBox_topology_plot_y_axis.currentText() == "Conduction Losses Transistor1 [W]":
                    vec_y_axis = converter.f_vec_conduction_losses1(zeta=vec_zeta,
                                                                    v_in=vec_v_in,
                                                                    v_out=vec_v_out,
                                                                    p_out=vec_output_power,
                                                                    v_g_on1=v_g_on1,
                                                                    transistor1=transistor1,
                                                                    transistor2=transistor2)
                if comboBox_topology_plot_y_axis.currentText() == "Conduction Losses Diode Transistor2 [W]":
                    vec_y_axis = converter.f_vec_conduction_losses2(zeta=vec_zeta,
                                                                    v_in=vec_v_in,
                                                                    v_out=vec_v_out,
                                                                    p_out=vec_output_power,
                                                                    v_g_on1=v_g_on1,
                                                                    transistor1=transistor1,
                                                                    transistor2=transistor2)
                if comboBox_topology_plot_y_axis.currentText() == "Total Conduction Losses [W]":
                    vec_y_axis = converter.f_vec_conduction_losses(zeta=vec_zeta,
                                                                   v_in=vec_v_in,
                                                                   v_out=vec_v_out,
                                                                   p_out=vec_output_power,
                                                                   v_g_on1=v_g_on1,
                                                                   transistor1=transistor1,
                                                                   transistor2=transistor2)
                if comboBox_topology_plot_y_axis.currentText() == "Turn-on Switching Losses Transistor1 [W]":
                    vec_y_axis = converter.f_vec_p_on1(zeta=vec_zeta,
                                                       v_in=vec_v_in,
                                                       v_out=vec_v_out,
                                                       p_out=vec_output_power,
                                                       v_g_on1=v_g_on1,
                                                       transistor1=transistor1,
                                                       transistor2=transistor2,
                                                       r_g_on1=r_g_on1,
                                                       frequency=vec_frequency)
                if comboBox_topology_plot_y_axis.currentText() == "Turn-off Switching Losses Transistor1 [W]":
                    vec_y_axis = converter.f_vec_p_off1(zeta=vec_zeta,
                                                        v_in=vec_v_in,
                                                        v_out=vec_v_out,
                                                        p_out=vec_output_power,
                                                        v_g_on1=v_g_on1,
                                                        transistor1=transistor1,
                                                        transistor2=transistor2,
                                                        r_g_off1=r_g_off1,
                                                        frequency=vec_frequency)
                if comboBox_topology_plot_y_axis.currentText() == "Reverse Recovery Losses Diode Transistor2 [W]":
                    vec_y_axis = converter.f_vec_p_rr2(zeta=vec_zeta,
                                                       v_in=vec_v_in,
                                                       v_out=vec_v_out,
                                                       p_out=vec_output_power,
                                                       v_g_on1=v_g_on1,
                                                       transistor1=transistor1,
                                                       transistor2=transistor2,
                                                       frequency=vec_frequency)
                if comboBox_topology_plot_y_axis.currentText() == "Total Switching Losses Transistor1 [W]":
                    vec_y_axis = converter.f_vec_p_on_off1(zeta=vec_zeta,
                                                              v_in=vec_v_in,
                                                              v_out=vec_v_out,
                                                              p_out=vec_output_power,
                                                              v_g_on1=v_g_on1,
                                                              transistor1=transistor1,
                                                              transistor2=transistor2,
                                                              r_g_on1=r_g_on1,
                                                              r_g_off1=r_g_off1,
                                                              frequency=vec_frequency)
                if comboBox_topology_plot_y_axis.currentText() == "Total Switching Losses [W]":
                    vec_y_axis = converter.f_vec_p_on_off_rr_1_2(zeta=vec_zeta,
                                                                 v_in=vec_v_in,
                                                                 v_out=vec_v_out,
                                                                 p_out=vec_output_power,
                                                                 v_g_on1=v_g_on1,
                                                                 transistor1=transistor1,
                                                                 transistor2=transistor2,
                                                                 r_g_on1=r_g_on1,
                                                                 r_g_off1=r_g_off1,
                                                                 frequency=vec_frequency)
                if comboBox_topology_plot_y_axis.currentText() == "Total Power Losses Transistor1 [W]":
                    vec_y_axis = converter.f_vec_p1(zeta=vec_zeta,
                                                    v_in=vec_v_in,
                                                    v_out=vec_v_out,
                                                    p_out=vec_output_power,
                                                    v_g_on1=v_g_on1,
                                                    transistor1=transistor1,
                                                    transistor2=transistor2,
                                                    r_g_on1=r_g_on1,
                                                    r_g_off1=r_g_off1,
                                                    frequency=vec_frequency)
                if comboBox_topology_plot_y_axis.currentText() == "Temperature Switch Transistor1 [°C]":
                    vec_y_axis = converter.f_vec_t_switch1(zeta=vec_zeta,
                                                           v_in=vec_v_in,
                                                           v_out=vec_v_out,
                                                           p_out=vec_output_power,
                                                           v_g_on1=v_g_on1,
                                                           r_g_on1=r_g_on1,
                                                           r_g_off1=r_g_off1,
                                                           t_heatsink=t_heatsink,
                                                           r_th_heatsink=r_th_heatsink,
                                                           frequency=vec_frequency,
                                                           transistor1=transistor1,
                                                           transistor2=transistor2)
                if comboBox_topology_plot_y_axis.currentText() == "Temperature Diode Transistor2 [°C]":
                    vec_y_axis = converter.f_vec_t_diode2(zeta=vec_zeta,
                                                          v_in=vec_v_in,
                                                          v_out=vec_v_out,
                                                          p_out=vec_output_power,
                                                          v_g_on1=v_g_on1,
                                                          t_heatsink=t_heatsink,
                                                          r_th_heatsink=r_th_heatsink,
                                                          frequency=vec_frequency,
                                                          transistor1=transistor1,
                                                          transistor2=transistor2)

                matplotlibwidget.axis.plot(vec_x_axis, vec_y_axis)
                matplotlibwidget.axis.ticklabel_format(useOffset=False)
                matplotlibwidget.axis.grid()
                matplotlibwidget.axis.set_position([0.15, 0.15, 0.8, 0.8])
                matplotlibwidget.axis.xaxis.set_major_formatter(FormatStrFormatter('%.2f'))
                matplotlibwidget.axis.yaxis.set_major_formatter(FormatStrFormatter('%.2f'))
                matplotlibwidget.axis.set(xlabel=comboBox_topology_plot_x_axis.currentText(),
                                          ylabel=comboBox_topology_plot_y_axis.currentText(), )
                matplotlibwidget.figure.canvas.draw_idle()

                matplotlibwidget.cursor = Cursor(matplotlibwidget.axis, horizOn=True, vertOn=True, useblit=True,
                                                 color="Green", linewidth=1)
                matplotlibwidget.figure.canvas.mpl_connect("button_press_event", clicked)

        except:
            matplotlibwidget.axis.clear()
            matplotlibwidget.figure.canvas.draw_idle()
            self.show_popup_message("Error: "+comboBox_topology_plot_y_axis.currentText()+" could not be plotted due to missing inputs or data!")





    def topology_update_plots(self):
        """
        Sets the variable to determine the topology and runs the function "topology_create_plot" for all QWidgets,
        Matplotlib figures and ComboBoxes

        :return: None
        """
        if self.comboBox_topology_topology.currentText() == "Buck-Converter":
            converter = buck_converter_functions
        elif self.comboBox_topology_topology.currentText() == "Boost-Converter":
            converter = boost_converter_functions
        elif self.comboBox_topology_topology.currentText() == "Buck-Boost-Converter":
            converter = buck_boost_converter_functions

        self.topology_create_plot(self.widget_topology_plot1,
                                  self.matplotlibwidget_topology1,
                                  self.comboBox_topology_plot1_x_axis,
                                  self.comboBox_topology_plot1_y_axis,
                                  self.comboBox_topology_plot1_z_axis,
                                  self.comboBox_topology_plot1_line_contour,
                                  converter)
        self.topology_create_plot(self.widget_topology_plot2,
                                  self.matplotlibwidget_topology2,
                                  self.comboBox_topology_plot2_x_axis,
                                  self.comboBox_topology_plot2_y_axis,
                                  self.comboBox_topology_plot2_z_axis,
                                  self.comboBox_topology_plot2_line_contour,
                                  converter)
        self.topology_create_plot(self.widget_topology_plot3,
                                  self.matplotlibwidget_topology3,
                                  self.comboBox_topology_plot3_x_axis,
                                  self.comboBox_topology_plot3_y_axis,
                                  self.comboBox_topology_plot3_z_axis,
                                  self.comboBox_topology_plot3_line_contour,
                                  converter)
        self.topology_create_plot(self.widget_topology_plot4,
                                  self.matplotlibwidget_topology4,
                                  self.comboBox_topology_plot4_x_axis,
                                  self.comboBox_topology_plot4_y_axis,
                                  self.comboBox_topology_plot4_z_axis,
                                  self.comboBox_topology_plot4_line_contour,
                                  converter)
        self.topology_create_plot(self.widget_topology_plot5,
                                  self.matplotlibwidget_topology5,
                                  self.comboBox_topology_plot5_x_axis,
                                  self.comboBox_topology_plot5_y_axis,
                                  self.comboBox_topology_plot5_z_axis,
                                  self.comboBox_topology_plot5_line_contour,
                                  converter)
        self.topology_create_plot(self.widget_topology_plot6,
                                  self.matplotlibwidget_topology6,
                                  self.comboBox_topology_plot6_x_axis,
                                  self.comboBox_topology_plot6_y_axis,
                                  self.comboBox_topology_plot6_z_axis,
                                  self.comboBox_topology_plot6_line_contour,
                                  converter)


    def get_converter(self):
        """
        Returns a converter_functions python file depending on which converter is currently selected

        :return: converter_functions python file
        """
        if self.comboBox_topology_topology.currentText() == "Buck-Converter":
            converter = buck_converter_functions
        elif self.comboBox_topology_topology.currentText() == "Boost-Converter":
            converter = boost_converter_functions
        elif self.comboBox_topology_topology.currentText() == "Buck-Boost-Converter":
            converter = buck_boost_converter_functions

        return converter


    def topology_pop_out_plot1(self):
        """
        Opens new window with plot1 from Topology Calculator

        :return:
        """
        converter = self.get_converter()

        self.PopOutPlotWindow1 = PopOutPlotWindow()
        self.matplotlibwidget_pop_out1 = MatplotlibWidget()


        self.topology_create_plot(self.PopOutPlotWindow1.widget_plot,
                                  self.matplotlibwidget_pop_out1,
                                  self.comboBox_topology_plot1_x_axis,
                                  self.comboBox_topology_plot1_y_axis,
                                  self.comboBox_topology_plot1_z_axis,
                                  self.comboBox_topology_plot1_line_contour,
                                  converter)
        self.PopOutPlotWindow1.show()

    def topology_pop_out_plot2(self):
        """
        Opens new window with plot2 from Topology Calculator

        :return:
        """
        converter = self.get_converter()

        self.PopOutPlotWindow2 = PopOutPlotWindow()
        self.matplotlibwidget_pop_out2 = MatplotlibWidget()


        self.topology_create_plot(self.PopOutPlotWindow2.widget_plot,
                                  self.matplotlibwidget_pop_out2,
                                  self.comboBox_topology_plot2_x_axis,
                                  self.comboBox_topology_plot2_y_axis,
                                  self.comboBox_topology_plot2_z_axis,
                                  self.comboBox_topology_plot2_line_contour,
                                  converter)
        self.PopOutPlotWindow2.show()

    def topology_pop_out_plot3(self):
        """
        Opens new window with plot3 from Topology Calculator

        :return:
        """
        converter = self.get_converter()

        self.PopOutPlotWindow3 = PopOutPlotWindow()
        self.matplotlibwidget_pop_out3 = MatplotlibWidget()


        self.topology_create_plot(self.PopOutPlotWindow3.widget_plot,
                                  self.matplotlibwidget_pop_out3,
                                  self.comboBox_topology_plot3_x_axis,
                                  self.comboBox_topology_plot3_y_axis,
                                  self.comboBox_topology_plot3_z_axis,
                                  self.comboBox_topology_plot3_line_contour,
                                  converter)
        self.PopOutPlotWindow3.show()

    def topology_pop_out_plot4(self):
        """
        Opens new window with plot4 from Topology Calculator

        :return:
        """
        converter = self.get_converter()

        self.PopOutPlotWindow4 = PopOutPlotWindow()
        self.matplotlibwidget_pop_out4 = MatplotlibWidget()


        self.topology_create_plot(self.PopOutPlotWindow4.widget_plot,
                                  self.matplotlibwidget_pop_out4,
                                  self.comboBox_topology_plot4_x_axis,
                                  self.comboBox_topology_plot4_y_axis,
                                  self.comboBox_topology_plot4_z_axis,
                                  self.comboBox_topology_plot4_line_contour,
                                  converter)
        self.PopOutPlotWindow4.show()

    def topology_pop_out_plot5(self):
        """
        Opens new window with plot5 from Topology Calculator

        :return:
        """
        converter = self.get_converter()

        self.PopOutPlotWindow5 = PopOutPlotWindow()
        self.matplotlibwidget_pop_out5 = MatplotlibWidget()


        self.topology_create_plot(self.PopOutPlotWindow5.widget_plot,
                                  self.matplotlibwidget_pop_out5,
                                  self.comboBox_topology_plot5_x_axis,
                                  self.comboBox_topology_plot5_y_axis,
                                  self.comboBox_topology_plot5_z_axis,
                                  self.comboBox_topology_plot5_line_contour,
                                  converter)
        self.PopOutPlotWindow5.show()

    def topology_pop_out_plot6(self):
        """
        Opens new window with plot6 from Topology Calculator

        :return:
        """
        converter = self.get_converter()

        self.PopOutPlotWindow6 = PopOutPlotWindow()
        self.matplotlibwidget_pop_out6 = MatplotlibWidget()


        self.topology_create_plot(self.PopOutPlotWindow6.widget_plot,
                                  self.matplotlibwidget_pop_out6,
                                  self.comboBox_topology_plot6_x_axis,
                                  self.comboBox_topology_plot6_y_axis,
                                  self.comboBox_topology_plot6_z_axis,
                                  self.comboBox_topology_plot6_line_contour,
                                  converter)
        self.PopOutPlotWindow6.show()

    def show_popup_message(self, message):
        """
        Pops up a notification window with a specific message

        :param message: notification message
        :return: None
        """
        MessageBox = QMessageBox()
        MessageBox.setWindowTitle("Information")
        MessageBox.setText(message)
        MessageBox.setIcon(QMessageBox.Warning)
        MessageBox.StandardButtons(QMessageBox.Cancel)
        MessageBox.exec_()



class CurveCheckerWindow(QMainWindow):
    """
    CurveCheckerWindow class which inherits from QMainWindow and contains all the code and functions to implement all
    functionalities into the code generated by QTDesigner in CurveCheckerWindow.ui
    """

    def __init__(self):
        super(CurveCheckerWindow, self).__init__()
        uic.loadUi("CurveCheckerWindow.ui", self)

        self.setWindowIcon(QtGui.QIcon(resource_path("window_icon")))

        self.matplotlibwidget = MatplotlibWidget()
        self.matplotlibwidget.axis_cm.remove()

        self.checkBox_first_xy_to_00.stateChanged.connect(self.update_curve)
        self.checkBox_second_y_to_0.stateChanged.connect(self.update_curve)
        self.checkBox_first_x_to_0.stateChanged.connect(self.update_curve)
        self.checkBox_mirror_xy_data.stateChanged.connect(self.update_curve)
        self.radioButton_scale_linear.toggled.connect(self.update_curve)
        self.radioButton_scale_log_x.toggled.connect(self.update_curve)
        self.radioButton_scale_log_y.toggled.connect(self.update_curve)
        self.radioButton_scale_log_xy.toggled.connect(self.update_curve)

        self.button_save_curve.clicked.connect(self.save_curve)

    def save_curve(self):
        """
        Saves curve to the corresponding comboBox and coverts the graph key to an array using the function tdb.csv2array()
        and then removes the file path key from the dict so that the dict can be used to create a transistor

        :return: None
        """
        curve_title = self.matplotlibwidget.axis.get_title()

        if curve_title == "Switch Channel Curve":
            comboBox = MainWindow.comboBox_create_transistor_switch_added_curves_channel_data
        elif curve_title == "Switch Switching Losses Curve":
            comboBox = MainWindow.comboBox_create_transistor_switch_added_curves_switching_losses
        elif curve_title == "Switch Gate Charge Curve":
            comboBox = MainWindow.comboBox_create_transistor_switch_added_curves_gate_charge
        elif curve_title == "Switch On Resistance Curve":
            comboBox = MainWindow.comboBox_create_transistor_switch_added_curves_r_on
        elif curve_title == "Switch SOA Curve":
            comboBox = MainWindow.comboBox_create_transistor_switch_added_curves_soa_t_pulse
        elif curve_title == "Diode Channel Curve":
            comboBox = MainWindow.comboBox_create_transistor_diode_added_curves_channel_data
        elif curve_title == "Diode Reverse Recovery Losses Curve":
            comboBox = MainWindow.comboBox_create_transistor_diode_added_curves_switching_losses
        elif curve_title == "Diode SOA Curve":
            comboBox = MainWindow.comboBox_create_transistor_diode_added_curves_soa_t_pulse
        elif curve_title == "Switch T_Rthjc Curve":
            comboBox = MainWindow.comboBox_create_transistor_switch_added_curve_t_rthjc
        elif curve_title == "Diode T_Rthjc Curve":
            comboBox = MainWindow.comboBox_create_transistor_diode_added_curve_t_rthjc
        elif curve_title == "E_coss Curve":
            comboBox = MainWindow.comboBox_create_transistor_added_curve_v_ecoss
        elif curve_title == "C_iss_normal Curve":
            comboBox = MainWindow.comboBox_create_transistor_added_c_iss_normal
        elif curve_title == "C_iss_detail Curve":
            comboBox = MainWindow.comboBox_create_transistor_added_c_iss_detail
        elif curve_title == "C_oss_normal Curve":
            comboBox = MainWindow.comboBox_create_transistor_added_c_oss_normal
        elif curve_title == "C_oss_detail Curve":
            comboBox = MainWindow.comboBox_create_transistor_added_c_oss_detail
        elif curve_title == "C_rss_normal Curve":
            comboBox = MainWindow.comboBox_create_transistor_added_c_rss_normal
        elif curve_title == "C_rss_detail Curve":
            comboBox = MainWindow.comboBox_create_transistor_added_c_rss_detail


        data_dict = self.comboBox_data.itemData(self.comboBox_data.currentIndex())

        for key in data_dict.keys():
            if "graph" in key:
                data_dict[key] = tdb.csv2array(data_dict["path"],
                              first_xy_to_00=self.checkBox_first_xy_to_00.isChecked(),
                              second_y_to_0=self.checkBox_second_y_to_0.isChecked(),
                              first_x_to_0=self.checkBox_first_x_to_0.isChecked(),
                              mirror_xy_data=self.checkBox_mirror_xy_data.isChecked())

        data_dict.pop("path")

        comboBox.setItemData(comboBox.currentIndex(), data_dict)
        comboBox.setDisabled(False)

        self.close()


    def update_curve(self):
        """
        Changes the curve when a CheckBox is checked in the CurveCheckerWindow using the function tdb.csv2array()

        :return: None
        """
        data_dict = self.comboBox_data.itemData(self.comboBox_data.currentIndex())
        graph = tdb.csv2array(data_dict["path"],
                              first_xy_to_00=self.checkBox_first_xy_to_00.isChecked(),
                              second_y_to_0=self.checkBox_second_y_to_0.isChecked(),
                              first_x_to_0=self.checkBox_first_x_to_0.isChecked(),
                              mirror_xy_data=self.checkBox_mirror_xy_data.isChecked())

        xlabel = self.matplotlibwidget.axis.get_xlabel()
        ylabel = self.matplotlibwidget.axis.get_ylabel()
        curve_title = self.matplotlibwidget.axis.get_title()
        curve_label = self.comboBox_data.currentText()
        self.matplotlibwidget.axis.clear()

        if self.radioButton_scale_log_xy.isChecked() == True:
            self.matplotlibwidget.axis.loglog(graph[0], graph[1], label=curve_label)
        elif self.radioButton_scale_linear.isChecked() == True:
            self.matplotlibwidget.axis.plot(graph[0], graph[1], label=curve_label)
        elif self.radioButton_scale_log_x.isChecked() == True:
            self.matplotlibwidget.axis.semilogx(graph[0], graph[1], label=curve_label)
        elif self.radioButton_scale_log_y.isChecked() == True:
            self.matplotlibwidget.axis.semilogy(graph[0], graph[1], label=curve_label)

        self.matplotlibwidget.axis.grid()
        self.matplotlibwidget.axis.set(xlabel=xlabel,
                                       ylabel=ylabel,
                                       title=curve_title)
        self.matplotlibwidget.axis.legend(fontsize=6)
        self.matplotlibwidget.figure.canvas.draw_idle()

        self.matplotlibwidget.cursor = Cursor(self.matplotlibwidget.axis, horizOn=True, vertOn=True, useblit=True,
                                         color="Green", linewidth=1)


    def run_curve_checker(self, comboBox, xlabel, ylabel, curve_title):
        """
        Runs the CurveChecker with a given comboBox containing an added curve

        :param comboBox: comboBox object that contains the added curve
        :param xlabel: xlabel for the added curve
        :param ylabel: ylabel for the added curve
        :param curve_title: curve_title for the added curve
        :return: None
        """
        try:
            self.layout = QVBoxLayout(self.widget_plot)
            self.layout.addWidget(self.matplotlibwidget)

            data_dict = comboBox.itemData(comboBox.currentIndex())
            curve_label = comboBox.currentText()

            self.comboBox_data.addItem(curve_label, data_dict)

            self.comboBox_data.setDisabled(True)

            self.matplotlibwidget.axis.set(xlabel=xlabel,
                                           ylabel=ylabel,
                                           title=curve_title)

            self.radioButton_scale_linear.setChecked(True)

            self.update_curve()

            self.show()
        except:
            comboBox.removeItem(comboBox.currentIndex())
            MainWindow.show_popup_message("Selected file is not valid!")


    def closeEvent(self, event):
        """
        Saves curve and gets window size settings when window is closed

        :param event: event signal when window is closed
        :return: None
        """
        self.save_curve()


class InformationWindow(QMainWindow):
    """
    InformationWindow class which inherits from QMainWindow and contains all the code and functions to implement all
    functionalities into the code generated by QTDesigner in InformationWindow.ui
    """
    def __init__(self):
        super(InformationWindow, self).__init__()
        uic.loadUi("InformationWindow.ui", self)
        self.setWindowIcon(QtGui.QIcon("window_icon"))

        self.button_yes.clicked.connect(self.button_yes_pressed)
        self.button_cancel.clicked.connect(self.button_cancel_pressed)

    def button_yes_pressed(self):
        """
        Runs the function to overwrite transistor, when the button "Yes" is pressed

        :return: None
        """
        MainWindow.overwrite_transistor()
        self.close()

    def button_cancel_pressed(self):
        """
        "Closes the information window the the button "Cancel" is pressed

        :return: None
        """

        MainWindow.button_create_transistor_create.setDisabled(True)

        self.close()

    def run_information_window(self):
        """
        Runs the information window

        :return: None
        """
        dict = MainWindow.create_transistor()
        transistor = dict["transistor"]

        self.label_text.setText(f"<b>{transistor.name}</b> already exists in the Local Database! Do you want to overwrite the existing transistor?")

        self.show()


class ViewCurveWindow(QMainWindow):
    """
    ViewCurveWindow class which inherits from QMainWindow and contains all the code and functions to implement all
    functionalities into the code generated by QTDesigner in ViewCurveWindow.ui
    """

    def __init__(self):
        super(ViewCurveWindow, self).__init__()
        uic.loadUi("ViewCurveWindow.ui", self)

        self.setWindowIcon(QtGui.QIcon("window_icon"))

        self.matplotlibwidget = MatplotlibWidget()
        self.matplotlibwidget.axis_cm.remove()

        self.radioButton_scale_linear.toggled.connect(self.update_curve)
        self.radioButton_scale_log_x.toggled.connect(self.update_curve)
        self.radioButton_scale_log_y.toggled.connect(self.update_curve)
        self.radioButton_scale_log_xy.toggled.connect(self.update_curve)

    def update_curve(self):
        """
        Updates the scaling of the curve when the radio buttons to scale linear/log(x)/log(y)/log(x,y) is pressed

        :return: None
        """
        graph = self.comboBox_data.itemData(self.comboBox_data.currentIndex())

        xlabel = self.matplotlibwidget.axis.get_xlabel()
        ylabel = self.matplotlibwidget.axis.get_ylabel()
        curve_title = self.matplotlibwidget.axis.get_title()
        curve_label = self.comboBox_data.currentText()
        self.matplotlibwidget.axis.clear()

        if self.radioButton_scale_log_xy.isChecked() == True:
            self.matplotlibwidget.axis.loglog(graph[0], graph[1], label=curve_label)
        elif self.radioButton_scale_linear.isChecked() == True:
            self.matplotlibwidget.axis.plot(graph[0], graph[1], label=curve_label)
        elif self.radioButton_scale_log_x.isChecked() == True:
            self.matplotlibwidget.axis.semilogx(graph[0], graph[1], label=curve_label)
        elif self.radioButton_scale_log_y.isChecked() == True:
            self.matplotlibwidget.axis.semilogy(graph[0], graph[1], label=curve_label)

        self.matplotlibwidget.axis.set(xlabel=xlabel,
                                       ylabel=ylabel,
                                       title=curve_title)

        self.matplotlibwidget.axis.legend(fontsize=6)
        self.matplotlibwidget.axis.grid()

        self.matplotlibwidget.figure.canvas.draw_idle()

        self.matplotlibwidget.cursor = Cursor(self.matplotlibwidget.axis, horizOn=True, vertOn=True, useblit=True,
                                         color="Green", linewidth=1)



    def view_curve(self, comboBox, curve_title, xlabel, ylabel):
        """
        Shows currently selected curve in a ViewCurveWindow

        :return: None
        """
        try:
            self.layout = QVBoxLayout(self.widget_plot)
            self.layout.addWidget(self.matplotlibwidget)

            data_dict = comboBox.itemData(comboBox.currentIndex())
            curve_label = comboBox.currentText()

            for key in data_dict.keys():
                if data_dict[key] is not None and "graph" in key:
                    graph = data_dict[key]


            self.comboBox_data.addItem(curve_label, graph)
            self.comboBox_data.setDisabled(True)

            self.matplotlibwidget.axis.set(xlabel=xlabel,
                                           ylabel=ylabel,
                                           title=curve_title)
            # set linear scale as default
            self.radioButton_scale_linear.setChecked(True)
            self.update_curve()

            self.show()
        except:
            pass


class PopOutPlotWindow(QMainWindow):
    """
    PopOutPlotWindow class which inherits from QMainWindow and contains all the code and functions to implement all
    functionalities into the code generated by QTDesigner in PopOutPlotWindow.ui
    """

    def __init__(self):
        super(PopOutPlotWindow, self).__init__()
        uic.loadUi("PopOutPlotWindow.ui", self)

        self.setWindowIcon(QtGui.QIcon("window_icon"))

        self.matplotlibwidget = MatplotlibWidget


class MatplotlibWidget(QWidget):
    """
    MatplotlibWidget class which inherits from QWidget and is used to implement a Matplotlib figure inside a QWidget
    """
    def __init__(self, parent=None):
        super(MatplotlibWidget, self).__init__(parent)
        self.figure = Figure()
        self.canvas = FigureCanvasQTAgg(self.figure)
        self.axis = self.figure.add_subplot(111)
        self.layout = QVBoxLayout(self)
        self.toolbar = NavigationToolbar2QT(self.canvas, self)
        self.layout.addWidget(self.canvas)
        self.layout.addWidget(self.toolbar)
        self.divider = make_axes_locatable(self.axis)
        self.axis_cm = self.divider.append_axes("right", size="3%", pad=0.03)
        self.sm = plt.cm.ScalarMappable(cmap=cm.inferno)
        self.figure.colorbar(mappable=self.sm, cax=self.axis_cm)


class EmittingStream(QtCore.QObject):
    """
    EmittingStream class to catch standard output and show it within the stdout-textEdit

    :return: None
    """
    text_written = QtCore.pyqtSignal(str)

    def write(self, text):
        """
        Writes standard output in stdout-textEdit

        :return: None
        """
        self.text_written.emit(str(text))


if __name__ == "__main__":
    import sys
    app = QApplication(sys.argv)
    MainWindow = MainWindow()
    MainWindow.show()
    sys.exit(app.exec_())
