# Copyright (c) 2021 cyb3rdog
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License in the file LICENSE.txt or at
#
#     https://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""
Serialization classes for the EscapePod Exntension Proxy SDK.
"""

# __all__ should order by constants, event classes, other classes, functions.
__all__ = ['keep_alive', 'process_intent', 'subscribed', 'escapepod_intent']


import json
from typing import Any, List
from .messaging import protocol


class proxy_message(object):
    def __init__(self, response: protocol.ProxyMessaage):
        self.message_type = protocol.MessageType.Name(response.message_type)
        self.message_data = response.message_data.replace("'", '"')


class process_intent(proxy_message):
    def __init__(self, response: protocol.ProxyMessaage):
        super().__init__(response)
        self.intent_name = response.intent_name
        json.loads(self.message_data, object_hook = self.deserialize)

    def deserialize(self, json: json):
        self.uuid = json['uuid']
        self.message = json['message']


class subscribed(proxy_message):
    def __init__(self, response: protocol.ProxyMessaage):
        super().__init__(response)
        json.loads(self.message_data, object_hook = self.deserialize)

    def deserialize(self, json: json):
        self.uuid = json['uuid']
        self.message = json['message']


class keep_alive(proxy_message):
    def __init__(self, response: protocol.ProxyMessaage):
        super().__init__(response)
        json.loads(self.message_data, object_hook = self.deserialize)

    def deserialize(self, json: json):
        self.timestamp = json['time'][0:json['time'].index(' ', 20)]


class escapepod_intent():

    class object_id():
        def __init__(self, _oid: str) -> None:
            self.oid = _oid

        def empty():
            return escapepod_intent.object_id("")

        def deserialize(data: json):
            return escapepod_intent.object_id(data['$oid'])

        def from_json(data: str):
            return json.loads(data, object_hook = escapepod_intent.object_id.deserialize)


    class options():
        def __init__(self, _external_parser, _block_list = None, _parser = None) -> None:
            self.external_parser = _external_parser
            self.block_list = _block_list
            self.parser = _parser

        def deserialize(data: json):
            # TODO: block_list & parser
            return escapepod_intent.options(data['external_parser'])
    
        def from_json(data: str):
            return escapepod_intent.options.deserialize(json.loads(data))


    def __init__(self, name: str, intent: str, utterance_list: str, description: str =None):
        self._json = None
        self._id = None
        self.name = name
        self.intent = intent
        self.utterance_list = utterance_list or ""
        self.description = description or "Intent generated by ExcapePod SDK for Python"

        self.extended_options = escapepod_intent.options(_external_parser = True)
        self.response_parameters = dict()
        self.response_parameters["final_intent"] = self.intent        

    @property 
    def id(self) -> object_id:
        return self._id or escapepod_intent.object_id.empty()

    @property 
    def json(self) -> str:
        return json.dumps(self._json, indent=2) or ""

    def deserialize(data: json):
        result = escapepod_intent(data['name'], data['intent'], data['utterance_list'], data['description'])
        result._json = data
        if not data['_id'] == None:
            result._id = escapepod_intent.object_id.from_json(json.dumps(data['_id']))
        else: result._id = escapepod_intent.object_id.empty()
        if not data['extended_options'] == None:
            result.extended_options = escapepod_intent.options.from_json(json.dumps(data['extended_options']))
        else: result.extended_options = None
        result.response_parameters = dict()
        if not data['response_parameters'] == None:
            for key in data['response_parameters'].keys():
                result.response_parameters[key] = data['response_parameters'][key]
        return result

    def from_json(data: str):
        return escapepod_intent.deserialize(json.loads(data))

    def serialize(self) -> str:
        return json.dumps(self, default=lambda o: o.__dict__, sort_keys=False, indent=-1)

    def to_json(self) -> str:
        result = "{"
        result += f"'intent':'{self.intent}',"
        result += f"'description':'{self.description}',"
        result += "'extended_options':{'external_parser':true,'block_list':null},"
        result += "'inversekeyphrases':null,"
        result += f"'name':'{self.name}',"
        result += f"'utterance_list':'{self.utterance_list}',"
        result += "'response_parameters':{" +f"'final_intent':'{self.intent}'" + "}"
        result += "}"
        return result.replace("'", '"')

    def to_string(self) -> str:
        return f"{self.intent}:\t{self.utterance_list}"

    def get_final_intent(self) -> str:
        return self.response_parameters["final_intent"] or self.intent

    def from_response(response: protocol.SelectIntentResponse) -> List[Any]:
        result = list()
        for key in response.intent_data.keys():
            intentJson = response.intent_data[key]
            result.append(escapepod_intent.from_json(intentJson))
        return result

    