
# from ..logger import CustomFormatter

import logging
import os

import logging
from etl.logger import CustomFormatter

logger = logging.getLogger(os.path.basename(__file__))
logger.setLevel(logging.DEBUG)
ch = logging.StreamHandler()
ch.setLevel(logging.DEBUG)
ch.setFormatter(CustomFormatter())
logger.addHandler(ch)


from sqlalchemy import create_engine,Column,Integer,String,Float, Date, DateTime, ForeignKey
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.orm import sessionmaker, relationship
from datetime import datetime

engine=create_engine('sqlite:///temp.db')

Base= declarative_base()

# Define the RatingsSmall table
class Rating(Base):
    
    """
    Representation of the Ratings_small table, storing user ratings of movies.

    Attributes
    ----------
    user_id : int
        The ID of the user who provided the rating.
    movie_id : int
        The ID of the movie that was rated.
    rating : float
        The rating given to the movie by the user.
    """
    __tablename__ = "Ratings_small"

    user_id = Column(Integer, ForeignKey('User.user_id'), primary_key=True)
    movie_id = Column(Integer, ForeignKey('MovieMetadata.movie_id'), primary_key=True)
    rating = Column(Float)

# Define the MovieMetadata table
class MovieMetadata(Base):
    """
    Representation of the MovieMetadata table, storing metadata about movies.

    Attributes
    ----------
    movie_id : int
        The primary key representing a unique ID for each movie.
    title : str
        The title of the movie.
    budget : float
        The budget of the movie.
    revenue : float
        The revenue generated by the movie.
    release_date : date
        The release date of the movie.
    language : str
        The language in which the movie was released.
    production_country : str
        The country where the movie was produced.
    production_company : str
        The company that produced the movie.
    """
    __tablename__ = "MovieMetadata"

    movie_id = Column(Integer, primary_key=True)
    title = Column(String)
    budget = Column(Float)
    revenue = Column(Float)
    release_date = Column(Date)
    language = Column(String)
    production_country = Column(String)
    production_company = Column(String)

    credits = relationship("Credits", back_populates="movie")
    keywords = relationship("Keywords", back_populates="movie")
    links_small = relationship("LinksSmall", back_populates="movie")

# Define the Credits table
class Credits(Base):
    """
    Representation of the Credits table, storing information about the cast and crew of movies.

    Attributes
    ----------
    movie_id : int
        The ID of the movie this entry relates to.
    cast : str
        A string representation of the main cast involved in the movie.
    crew : str
        A string representation of the main crew involved in the making of the movie.
    """
    __tablename__ = "Credits"

    movie_id = Column(Integer, ForeignKey('MovieMetadata.movie_id'), primary_key=True)
    cast = Column(String)  # Assuming cast is a string field
    crew = Column(String)  # Assuming crew is a string field

    movie = relationship("MovieMetadata", back_populates="credits")

# Define the Keywords table
class Keywords(Base):
    """
    Representation of the plot_keywords table, storing keywords associated with the plot of movies.

    Attributes
    ----------
    movie_id : int
        The ID of the movie this entry relates to.
    plot_keywords : str
        A string of comma-separated keywords that describe the plot of the movie.
    """
    __tablename__ = "plot_keywords"

    movie_id = Column(Integer, ForeignKey('MovieMetadata.movie_id'), primary_key=True)
    plot_keywords = Column(String)
    movie = relationship("MovieMetadata", back_populates="keywords")

class User(Base):
    """
    Representation of the User table, storing information about users.

    Attributes
    ----------
    user_id : int
        The primary key representing a unique ID for each user.
    first_name : str
        The first name of the user.
    last_name : str
        The last name of the user.
    email : str
        The email address of the user.
    age : int
        The age of the user.
    """
    __tablename__ = "User"

    user_id = Column(Integer, primary_key=True)
    first_name = Column(String)
    last_name = Column(String)
    email = Column(String)
    age = Column(Integer)
    
    # Define other user-related attributes here

    ratings = relationship("RatingsSmall")


class InOut(Base):
    """
    Representation of the InOut table, storing input movie names and corresponding output recommendations.

    Attributes
    ----------
    movie_id : int
        The primary key, representing the unique ID of the input movie.
    input_movie : str
        The name of the input movie.
    output_recommendations : str
        The recommended movies based on the input movie, stored as a string.
    """
    __tablename__ = "InOut"
    movie_id = Column(Integer, ForeignKey('MovieMetadata.movie_id'), primary_key=True)
    input_movie = Column(String)
    output_recommendations = Column(String)

    # Relationship to MovieMetadata
    movie = relationship("MovieMetadata")

Base.metadata.create_all(engine)